<?php
/**
 * Extend attachment endpoint.
 *
 * @package Nisje
 */

$taxonomy_format_name = nisje_get_setting( 'taxonomy_format_name', 'user-content' );
$taxonomy_type_name   = nisje_get_setting( 'taxonomy_type_name', 'user-content' );

// Global Nisje core tags.
$taxonomy_tags_name = 'nisje-tags';

$field_key = nisje_get_setting( 'field_key', 'event' );

$extend_object = [
	'post',
];

// Group ID.
register_rest_field(
	$extend_object, 'group_id', [
		'get_callback' => function ( $object ) {
			$ret_val  = null;
			$group_id = get_post_meta( $object['id'], '_nisje_group_connection', true );

			if ( $group_id ) {
				$ret_val = intval( $group_id );
			}

			return $ret_val;
		},
		'schema'       => [
			'context'     => [ 'view' ],
			'description' => esc_html__( 'The ID of the group associated with this user content post.', 'nisje-advanced-search' ),
			'type'        => 'integer',
			'readonly'    => false,
		],
	]
);

// Group name.
register_rest_field(
	$extend_object, 'group_name', [
		'get_callback' => function ( $object ) {
			$ret_val  = '';
			$group_id = get_post_meta( $object['id'], '_nisje_group_connection', true );
			$group    = nisje_get_group( $group_id );

			if ( ! is_wp_error( $group ) ) {
				$ret_val = bp_get_group_name( $group );
			}

			return $ret_val;
		},
		'schema'       => [
			'context'     => [ 'view' ],
			'description' => esc_html__( 'The name of the group associated with this user content post.', 'nisje-advanced-search' ),
			'type'        => 'string',
			'readonly'    => true,
		],
	]
);

// Attachments.
register_rest_field(
	$extend_object, 'attachments', [
		'get_callback'    => 'nisje_get_post_attachments',
		'update_callback' => 'nisje_update_post_attachments',
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'IDs of attachments', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'object',
			],
			'readonly'    => false,
		],
	]
);

// Images.
register_rest_field(
	$extend_object, 'images', [
		'get_callback'    => function ( $object ) {
			$retval = [];

			$images = get_post_meta( $object['id'], 'nisje-images', true );
			if ( ! empty( $images ) && is_array( $images ) ) {
				foreach ( $images as $image ) {
					$retval[] = nisje_populate_image_data( $image );
				}
			}

			return $retval;
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_format_name ) {
			if ( $object instanceof \WP_Post && ! empty( $field ) ) {
				update_post_meta( $object->ID, 'nisje-images', $field );

				wp_set_post_terms( $object->ID, [ 'image', 'gallery' ], $taxonomy_format_name );
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Array of image src', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => false,
		],
	]
);

// Share.
register_rest_field(
	$extend_object, 'share', [
		'get_callback'    => function ( $object ) {
			$share = get_post_meta( $object['id'], 'nisje-share', true );
			if ( $share && is_array( $share ) ) {
				return $share;
			}

			return [];
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_format_name ) {
			if ( $object instanceof \WP_Post && ( ! empty( $field ) && is_array( $field ) ) ) {

				$share = [];

				if ( isset( $field['url'] ) ) {
					$share['url'] = esc_url( $field['url'] );
				}

				if ( isset( $field['title'] ) ) {
					$share['title'] = $field['title'];
				}

				if ( isset( $field['description'] ) ) {
					$share['description'] = $field['description'];
				}

				update_post_meta( $object->ID, 'nisje-share', $share );

				wp_set_post_terms( $object->ID, 'link', $taxonomy_format_name );
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Meta data from shared url', 'nisje-advanced-search' ),
			'type'        => 'object',
			'properties'  => [
				'url'   => [
					'context'     => [ 'view', 'edit' ],
					'description' => esc_html__( 'Url to link', 'nisje-advanced-search' ),
					'type'        => 'string',
					'format'      => 'uri',
				],
				'title' => [
					'context'     => [ 'view', 'edit' ],
					'description' => esc_html__( 'Title to link', 'nisje-advanced-search' ),
					'type'        => 'string',
				],
			],
		],
	]
);

// Video.
register_rest_field(
	$extend_object, 'video', [
		'get_callback'    => function ( $object ) {
			$video = get_post_meta( $object['id'], 'nisje-video', true );
			if ( $video ) {
				return esc_url( $video );
			}

			return '';
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_format_name ) {
			if ( $object instanceof \WP_Post ) {
				update_post_meta( $object->ID, 'nisje-video', $field );

				wp_set_post_terms( $object->ID, 'video', $taxonomy_format_name );
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Video', 'nisje-advanced-search' ),
			'type'        => 'string',
			'format'      => 'url',
			'readonly'    => false,
		],
	]
);

// Add avatar URLs.
register_rest_field(
	$extend_object, 'avatar_urls', [
		'get_callback' => function ( $object ) {
			$post = get_post( $object['id'] );
			if ( $post instanceof \WP_Post ) {
				return nisje_get_user_avatar( $post->post_author );
			}

			return [];
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Avatar URLs for the resource', 'nisje-advanced-search' ),
			'type'        => 'object',
			'properties'  => nisje_get_avatar_rest_properties(),
			'readonly'    => true,
		],
	]
);

// Comment count.
register_rest_field(
	$extend_object, 'comment_count', [
		'get_callback' => function ( $object ) {
			$comments = wp_count_comments( $object['id'] );

			return $comments->total_comments;
		},
		'schema'       => [
			'context'     => [ 'view' ],
			'description' => esc_html__( 'Number of comments', 'nisje-advanced-search' ),
			'type'        => 'integer',
			'readonly'    => true,
		],
	]
);

// Comments.
register_rest_field(
	$extend_object, 'comments', [
		'get_callback' => function ( $object ) {
			$request = new WP_REST_Request( 'GET', '/wp/v2/comments' );

			$request->set_query_params(
				[
					'post' => $object['id'],
				]
			);

			$response = rest_do_request( $request );
			$server   = rest_get_server();
			$data     = $server->response_to_data( $response, false );

			return array_reverse( $data, false );
		},
		'schema'       => [
			'context'     => [ 'view' ],
			'description' => esc_html__( 'Comments object', 'nisje-advanced-search' ),
			'type'        => 'object',
			'readonly'    => true,
		],
	]
);

// Author.
register_rest_field(
	$extend_object, 'author', [
		'get_callback' => function ( $object ) {
			$retval = [
				'name' => '',
				'id'   => '',
			];

			$post = get_post( $object['id'] );
			if ( $post instanceof \WP_Post ) {
				$retval = [
					'name' => bp_core_get_user_displayname( $post->post_author ),
					'id'   => absint( $post->post_author ),
				];
			}

			return $retval;
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Author', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'string',
			],
			'readonly'    => true,
		],
	]
);

// Image data.
register_rest_field(
	$extend_object, 'image', [
		'get_callback' => function ( $object ) {
			if ( has_post_thumbnail( $object['id'] ) ) {
				return nisje_populate_image_data( get_post_thumbnail_id( $object['id'] ) );
			}

			return [];
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Image data', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'string',
			],
			'readonly'    => true,
		],
	]
);

// Formats.
register_rest_field(
	$extend_object, 'formats', [
		'get_callback' => function ( $object ) use ( $taxonomy_format_name ) {
			$formats = [];

			$terms = get_the_terms( $object['id'], $taxonomy_format_name );
			if ( $terms ) {
				foreach ( $terms as $term ) {
					$formats[ $term->term_id ] = $term->name;
				}
			}

			return $formats;
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Formats', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'string',
			],
			'readonly'    => true,
		],
	]
);

// Types.
register_rest_field(
	$extend_object, 'type', [
		'get_callback'    => function ( $object ) use ( $taxonomy_type_name ) {
			$type = [];

			$terms = get_the_terms( $object['id'], $taxonomy_type_name );
			if ( $terms ) {
				$type = $terms[0]->slug;
			} else {
				$post = get_post( $object['id'] );
				if ( $post instanceof \WP_Post ) {
					$type = $post->post_type;
				}
			}

			return $type;
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_type_name ) {
			if ( $object instanceof \WP_Post ) {
				wp_set_post_terms( $object->ID, $field, $taxonomy_type_name );
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'The content type', 'nisje-advanced-search' ),
			'type'        => 'string',
			'enum'        => array_keys( nisje_get_setting( 'types', 'user-content' ) ),
			'required'    => true,
		],
	]
);

// Types.
register_rest_field(
	$extend_object, 'tags', [
		'get_callback'    => function ( $object ) use ( $taxonomy_tags_name ) {
			$tags = [];

			$terms = get_the_terms( $object['id'], $taxonomy_tags_name );
			if ( $terms ) {
				foreach ( $terms as $term ) {
					$tags[ $term->term_id ] = $term;
				}
			}

			return $tags;
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_tags_name ) {
			if ( $object instanceof \WP_Post ) {
				wp_set_post_terms( $object->ID, $field, $taxonomy_tags_name );
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'The content tags', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'string',
			],
			'required'    => true,
		],
	]
);

// Reactions.
register_rest_field(
	$extend_object, 'reactions', [
		'get_callback' => function ( $object ) {
			return nisje_get_user_reactions( $object['id'], 'post' );
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Reactions', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'mixed',
			],
			'readonly'    => true,
		],
	]
);

// Add start date.
register_rest_field(
	$extend_object, 'start_date', [
		'get_callback'    => function ( $object ) use ( $field_key ) {
			return get_post_meta( $object['id'], $field_key . '_start_date', true );
		},
		'update_callback' => function ( $value, $object ) use ( $field_key ) {
			update_post_meta( $object->ID, $field_key . '_start_date', $value );

			return $value;
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Start date', 'nisje-advanced-search' ),
			'type'        => 'string',
			'format'      => 'date-time',
			'required'    => true,
		],
	]
);

// Add end date.
register_rest_field(
	$extend_object, 'end_date', [
		'get_callback'    => function( $object ) use ( $field_key ) {
			return get_post_meta( $object['id'], $field_key . '_end_date', true );
		},
		'update_callback' => function( $value, $object ) use ( $field_key ) {
			update_post_meta( $object->ID, $field_key . '_end_date', $value );

			return $value;
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'End date', 'nisje-advanced-search' ),
			'type'        => 'string',
			'format'      => 'date-time',
		],
	]
);

// Add location.
register_rest_field(
	$extend_object, 'location', [
		'get_callback'    => function( $object ) use ( $field_key ) {
			return esc_html( get_post_meta( $object['id'], $field_key . '_location_text', true ) );
		},
		'update_callback' => function( $value, $object ) use ( $field_key ) {
			$value = esc_html( $value );
			update_post_meta( $object->ID, $field_key . '_location_text', $value );

			return $value;
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Location', 'nisje-advanced-search' ),
			'type'        => 'string',
		],
	]
);

// Add Attending.
register_rest_field(
	$extend_object, 'attending', [
		'get_callback' => function( $object ) {
			return nisje_event_get_users_attending( $object['id'] );
		},
		'schema'       => [
			'description' => esc_html__( 'Users ids of users that is attending to this event.', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => true,
		],
	]
);

// Add Not Attending.
register_rest_field(
	$extend_object, 'not_attending', [
		'get_callback' => function( $object ) {
			return nisje_event_get_users_not_attending( $object['id'] );
		},
		'schema'       => [
			'description' => esc_html__( 'Users ids of users that is not attending to this event.', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => true,
		],
	]
);

// Add Maybe Attending.
register_rest_field(
	$extend_object, 'maybe_attending', [
		'get_callback' => function( $object ) {
			return nisje_event_get_users_maybe_attending( $object['id'] );
		},
		'schema'       => [
			'description' => esc_html__( 'Users ids of users that is maybe attending to this event.', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => true,
		],
	]
);

// Add Maybe Attending.
register_rest_field(
	$extend_object, 'invited', [
		'get_callback' => function( $object ) {
			return nisje_event_get_users_invited( $object['id'] );
		},
		'schema'       => [
			'description' => esc_html__( 'Users ids of users that is invited to this event.', 'nisje-advanced-search' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => true,
		],
	]
);
