<?php
/**
 * Setup Alerts Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Alerts Component Class
 */
class Alerts extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		$path = trailingslashit( __DIR__ );

		parent::start(
			'alerts',
			esc_html__( 'Alerts', 'nisje' ),
			$path
		);
	}

	/**
	 * Global variables
	 */
	public function setup_globals() {
		$defaults = [
			'post_type_name'   => 'nisje-alert',
			'capability_key'   => 'nisje_alert',
			'transient_key'    => 'nisje_alert_cache',
			'field_key_prefix' => 'nisje_alert',
		];

		$this->settings = wp_parse_args( $this->settings, $defaults );
	}

	/**
	 * Setup filters
	 */
	public function setup_filters() {
		add_filter( 'nisje_frontend_options', [ $this, 'add_frontend_option' ] );

		if ( is_admin() ) {
			$post_type = $this->settings['post_type_name'];

			add_action( 'nisje_register_acf_fields', [ $this, 'register_field_groups' ] );
			add_action( "save_post_{$post_type}", [ $this, 'flush_transient_cache' ] );
		}
	}

	/**
	 * Add frontend options
	 *
	 * @param array $options Options.
	 */
	public function add_frontend_option( array $options ): array {
		$transient_key = nisje_get_setting( 'transient_key', 'alerts' );

		$retval = get_transient( $transient_key );

		if ( false === $retval ) {
			$retval = [];

			$field_key_prefix = nisje_get_setting( 'field_key_prefix', 'alerts' );

			$args = [
				'post_type'      => nisje_get_setting( 'post_type_name', 'alerts' ),
				'posts_per_page' => 1,
				'post_status'    => 'publish',
			];

			$alerts = new \WP_Query( $args );
			if ( $alerts->have_posts() ) {
				while ( $alerts->have_posts() ) {
					$alerts->the_post();

					$color = get_post_meta( get_the_ID(), $field_key_prefix . '_color', true );

					if ( ! $color ) {
						$color = 'black';
					}

					$retval[] = [
						'id'      => get_the_ID(),
						'title'   => get_the_title(),
						'excerpt' => get_the_excerpt(),
						'color'   => esc_attr( $color ),
					];
				}
			}

			set_transient( $transient_key, $retval, 12 * HOUR_IN_SECONDS );
		}

		$options['alerts'] = $retval;

		return $options;
	}

	/**
	 * Include ACF field groups
	 */
	public function register_field_groups() {
		$this->include_file( 'acf/field-groups/acf-field-group-alert.php' );
	}


	/**
	 * Register Content Types
	 */
	public function register_content_types() {
		$this->include_file( 'post-types/post-type-alert.php' );
	}

	/**
	 * Delete transient if there is a post saved in the backend.
	 *
	 * @param int $post_id Post ID.
	 */
	public function flush_transient_cache( $post_id ) {
		$transient_key = nisje_get_setting( 'transient_key', 'alerts' );
		delete_transient( $transient_key );
	}
}
