<?php
/**
 * Set up blogs in groups component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Blogs in groups component class
 */
class BlogsInGroups extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		$path = trailingslashit( dirname( __FILE__ ) );

		parent::start(
			'blogs-in-groups',
			__( 'Blogs in Groups', 'nisje-blogs-in-groups' ),
			$path,
			'blogs'
		);
	}

	/**
	 * Global variables
	 */
	public function setup_globals() {
		$this->field_key = 'nisje_blogs';
		$this->key       = 'blogs';

		$defaults = [
			// Group.
			'post_type_name_group'          => 'nisje-blogs-group',
			'post_type_rest_endpoint_group' => 'nisje-blogs-group',
			'capability_key_group'          => 'nisje_blogs_cap_group',
			'activity_type_group'           => 'nisje_blogs_group',
			// General.
			'field_key'                     => $this->field_key,
			'key'                           => $this->key,
		];

		$this->settings = wp_parse_args( $this->settings, $defaults );
	}

	/**
	 * Add frontend options
	 *
	 * @param array $options Options.
	 */
	public function add_frontend_option( $options ) {
		$options['features']['groupBlogs'] = true;
		return $options;
	}

	/**
	 * Set up filters and actions
	 */
	public function setup_filters() {
		add_filter( 'nisje_rest_shared_fields_group_post_types', [ $this, 'add_shared_rest_group_fields' ] );
		add_filter( 'nisje_rest_shared_fields_post_types', [ $this, 'add_shared_rest_fields' ] );
		add_filter( 'nisje_feed_post_types', [ $this, 'add_feed_post_types' ] );
		add_action( 'nisje_set_group_meta', [ $this, 'set_group_meta' ], 10, 3 );
		add_action( 'nisje_update_group_meta', [ $this, 'update_group_meta' ], 10, 3 );
		add_action( 'nisje_add_to_prepared_group_data', [ $this, 'add_to_prepared_group_data' ], 10, 3 );
		add_filter( 'nisje_filter_group_data_object', [ $this, 'add_to_prepared_group_data_object' ], 10, 3 );
		add_filter( 'nisje_filter_group_schema_object', [ $this, 'add_to_group_schema_object' ], 10, 3 );
		add_filter( 'nisje_frontend_options', [ $this, 'add_frontend_option' ] );
		add_filter( 'nisje_group_features', [ $this, 'add_group_feature' ] );
	}

	/**
	 * Extend rest group types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_shared_rest_group_fields( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_rest_endpoint_group', $this->settings_key );

		return $post_types;
	}

	/**
	 * Extend rest post types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_shared_rest_fields( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_rest_endpoint_group', $this->settings_key );

		return $post_types;
	}

	/**
	 * Add feed post types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_feed_post_types( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_name_group', $this->settings_key );

		return $post_types;
	}

	/**
	 * Set group meta
	 *
	 * @param object          $group      Group.
	 * @param WP_REST_Request $request    Request object.
	 * @param array           $group_args Group Arguments.
	 */
	public function set_group_meta( $group, $request, $group_args ) {
		// Set Group Blog if active.
		if ( isset( $group_args['enable_blogs'] ) && $group_args['enable_blogs'] ) {
			groups_update_groupmeta( $group->id, '_nisje_blogs_enabled', true );
		}
	}

	/**
	 * Update group meta
	 *
	 * @param object          $group      Group.
	 * @param WP_REST_Request $request    Request object.
	 * @param array           $group_args Group Arguments.
	 */
	public function update_group_meta( $group, $request, $group_args ) {
		if ( isset( $group_args->enable_blogs ) ) {
			groups_update_groupmeta( $group->id, '_nisje_blogs_enabled', $group_args->enable_blogs );
		}
	}

	/**
	 * Add to prepared group data
	 *
	 * @param object          $schema         Group schema object.
	 * @param stdClass        $prepared_group An object representing a single post prepared for inserting or updating the database.
	 * @param WP_REST_Request $request        Request object.
	 */
	public function add_to_prepared_group_data( $schema, $prepared_group, $request ) {
		if ( ! empty( $schema['properties']['enable_blogs'] ) && isset( $request['enable_blogs'] ) ) {
			$prepared_group->enable_blogs = (bool) $request['enable_blogs'];
		}
	}

	/**
	 * Add to prepared group data object
	 *
	 * @param object $data Group prepared data object.
	 * @param object $schema Group schema object.
	 * @param array  $item Object.
	 *
	 * @return object $data Group prepared data object.
	 */
	public function add_to_prepared_group_data_object( $data, $schema, $item ) {
		if ( ! empty( $schema['properties']['enable_blogs'] ) ) {
			$data['enable_blogs'] = (bool) groups_get_groupmeta( $item->id, '_nisje_blogs_enabled', true );
		}

		return $data;
	}

	/**
	 * Add to group schema object
	 *
	 * @param object $schema Group schema object.
	 *
	 * @return object $schema Group schema object.
	 */
	public function add_to_group_schema_object( $schema ) {
		$additional_fields = [
			'properties' => [
				'enable_blogs' => [
					'context'     => [ 'view', 'edit', 'member' ],
					'description' => esc_html__( 'Whether the group has a blog or not.', 'nisje-blogs-in-groups' ),
					'type'        => 'boolean',
				],
			],
		];

		return array_replace_recursive( $schema, $additional_fields );
	}

	/**
	 * Add group features
	 *
	 * @param array $features Features.
	 */
	public function add_group_feature( $features ) {
		$features['enable_blogs'] = [
			'label'       => __( 'Blog', 'nisje-blogs-in-groups' ),
			'description' => __( 'Enable blogs for this group', 'nisje-blogs-in-groups' ),
		];

		return $features;
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		if ( bp_is_active( 'groups' ) ) {
			$this->include_file( 'rest-handlers/extend-groups.php' );
		}
	}

	/**
	 * Register content types
	 */
	public function register_content_types() {
		require dirname( __FILE__ ) . '/post-types/post-type-blogs-group.php';
	}
}
