<?php
/**
 * Add lang attributes on blocks
 *
 * @package nisje
 */

declare( strict_types=1 );
namespace Nisje\CSSR\Gutenberg;

/**
 * Get language attributes
 *
 * @param string $content Content.
 */
function get_attr( string $content ): string {
	$lang = \nisje_get_content_language( $content );
	$dir  = 'ar' === $lang ? 'rtl' : 'ltr';

	return 'lang="' . $lang . '" dir="' . $dir . '"';
}

/**
 * Optimize language
 *
 * @param string $block_content The block content about to be appended.
 * @param array  $block         The full block, including name and attributes.
 */
function render_block( string $block_content, array $block ): string {
	if ( ! function_exists( 'nisje_get_content_language' ) ) {
		return $block_content;
	}

	if ( 'core/paragraph' === $block['blockName'] ) {
		return preg_replace( '/(<p\b[^><]*)>/i', '$1 ' . get_attr( $block_content ) . '">', $block_content );
	}

	if ( 'core/heading' === $block['blockName'] ) {
		return preg_replace( '/(<h[1-6]\b[^><]*)>/i', '$1 ' . get_attr( $block_content ) . '">', $block_content );
	}

	return $block_content;
}
\add_filter( 'render_block', __NAMESPACE__ . '\\render_block', 10, 2 );

/**
 * Optimize title lang
 */
add_action( 'rest_api_init', function() {
	$post_types = apply_filters( 'nisje_rest_shared_fields_post_types', [] );

	if ( ! empty( $post_types ) && is_array( $post_types ) ) {
		\register_rest_field(
			$post_types, 'titleLang', [
				'get_callback' => function ( $object ) {
					if ( ! isset( $object['title']['raw'] ) ) {
						return '';
					}

					return \nisje_get_content_language( $object['title']['raw'] );
				},
				'schema'       => [
					'context'     => [ 'view', 'edit' ],
					'description' => esc_html__( 'Language of title', 'nisje' ),
					'type'        => 'string',
					'readonly'    => true,
				],
			]
		);
	}
} );
