<?php
/**
 * Event attend class.
 * Handles populating and saving users attending events.
 *
 * @TODO Add cache functionality for direct db calls.
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

// phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery

/**
 * Event attend class
 *
 * Handles populating and saving user attending
 */
class Nisje_Event_Attend {
	/**
	 * The attending ID.
	 *
	 * @var int
	 */
	public $id = 0;

	/**
	 * The event_id
	 *
	 * @var int
	 */
	public $event_id;

	/**
	 * The user_id
	 *
	 * @var int
	 */
	public $user_id;

	/**
	 * The inviter_id
	 *
	 * @var int
	 */
	public $inviter_id;

	/**
	 * The status
	 *
	 * @var string
	 */
	public $status;

	/**
	 * The date registered
	 *
	 * @var int
	 */
	public $date;

	/**
	 * Constructor
	 *
	 * @param int    $event_id   The event ID.
	 * @param int    $user_id    The user ID.
	 * @param string $status     Status.
	 * @param int    $inviter_id Inviter ID.
	 */
	public function __construct( $event_id = 0, $user_id = 0, $status = '', $inviter_id = 0 ) {
		if ( ! empty( $event_id ) && ! empty( $user_id ) && ! empty( $status ) ) {
			$this->event_id   = (int) $event_id;
			$this->user_id    = (int) $user_id;
			$this->inviter_id = (int) $inviter_id;
			$this->status     = sanitize_key( $status );

			$this->populate();
		}
	}

	/**
	 * Populate method.
	 *
	 * Used in constructor.
	 */
	protected function populate() {
		global $wpdb, $bp;

		$attend_id = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$wpdb->di_event_attending} WHERE event_id = %d AND user_id = %d", $this->event_id, $this->user_id ) );
		if ( $attend_id ) {
			$this->id = $attend_id;
		}
	}

	/**
	 * Saves attending
	 */
	public function save() {
		global $wpdb;

		if ( empty( $this->event_id ) || empty( $this->user_id ) || empty( $this->status ) ) {
			return false;
		}

		// Update existing entry.
		if ( $this->id ) {
			$result = $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->di_event_attending} SET event_id = %d, user_id = %d, inviter_id = %d, status = %s, date_modified = %s WHERE id = %d", $this->event_id, $this->user_id, $this->inviter_id, $this->status, current_time( 'mysql' ), $this->id ) );
		} else {
			$result   = $wpdb->query( $wpdb->prepare( "INSERT INTO {$wpdb->di_event_attending} ( event_id, user_id, inviter_id, status, date_created ) VALUES ( %d, %d, %d, %s, %s )", $this->event_id, $this->user_id, $this->inviter_id, $this->status, current_time( 'mysql' ) ) );
			$this->id = $wpdb->insert_id;
		}

		return $result;
	}

	/**
	 * Deletes from database
	 */
	public function delete() {
		global $wpdb;

		return $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->di_event_attending} WHERE id = %d", $this->id ) );
	}

	/**
	 * Get events for specific user
	 *
	 * @param int $user_id The user ID.
	 * @return array
	 */
	public static function get_events_conntected_to_user( $user_id ) {
		global $wpdb;

		$attends = $wpdb->get_results( $wpdb->prepare( "SELECT event_id, status FROM {$wpdb->di_event_attending} WHERE user_id = %d", $user_id ) );

		return [
			'attends' => $attends,
			'total'   => count( $attends ),
		];
	}

	/**
	 * Get users for a event
	 *
	 * @param int $event_id The event ID.
	 * @return array
	 */
	public static function get_users_connected_to_event( $event_id ) {
		global $wpdb;

		$attends = $wpdb->get_results( $wpdb->prepare( "SELECT user_id, status FROM {$wpdb->di_event_attending} WHERE event_id = %d", $event_id ) );

		return [
			'attends' => $attends,
			'total'   => count( $attends ),
		];
	}

	/**
	 * Get attending users for a specific event.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_attending( $event_id ) {
		global $wpdb;

		$user_ids = $wpdb->get_col( $wpdb->prepare( "SELECT user_id FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'attend'", $event_id ) );

		$display_names = [];

		foreach ( $user_ids as $user_id ) {
			$user = get_userdata( $user_id );

			$display_names[] = [
				'name' => $user->display_name,
				'id'   => (int) $user_id,
			];
		}

		return [
			'users' => $display_names,
			'total' => count( $user_ids ),
		];
	}

	/**
	 * Get not attending users for a specific event.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_not_attending( $event_id ) {
		global $wpdb;

		$user_ids = $wpdb->get_col( $wpdb->prepare( "SELECT user_id FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'not'", $event_id ) );

		$display_names = [];

		foreach ( $user_ids as $user_id ) {
			$user = get_userdata( $user_id );

			$display_names[] = [
				'name' => $user->display_name,
				'id'   => (int) $user_id,
			];
		}

		return [
			'users' => $display_names,
			'total' => count( $user_ids ),
		];
	}

	/**
	 * Get maybe attending users for a specific event.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_maybe_attending( $event_id ) {
		global $wpdb;

		$user_ids = $wpdb->get_col( $wpdb->prepare( "SELECT user_id FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'maybe'", $event_id ) );

		$display_names = [];

		foreach ( $user_ids as $user_id ) {
			$user = get_userdata( $user_id );

			$display_names[] = [
				'name' => $user->display_name,
				'id'   => (int) $user_id,
			];
		}

		return [
			'users' => $display_names,
			'total' => count( $user_ids ),
		];
	}

	/**
	 * Get invited users for a specific event.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_invited( $event_id ) {
		global $wpdb;

		$user_ids = $wpdb->get_col( $wpdb->prepare( "SELECT user_id FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'invite'", $event_id ) );

		$ids = [];

		foreach ( $user_ids as $user_id ) {
			$ids[] = (int) $user_id;
		}

		return $ids;
	}

	/**
	 * Get the number of events user is attending.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_attending_count( $event_id ) {
		global $wpdb;

		return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'attend'", $event_id ) );
	}

	/**
	 * Get the number of users attending to an event.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_not_attending_count( $event_id ) {
		global $wpdb;

		return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'not'", $event_id ) );
	}

	/**
	 * Get the number of events user is not attending.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_maybe_attending_count( $event_id ) {
		global $wpdb;

		return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'maybe'", $event_id ) );
	}

	/**
	 * Get the number of events user is invited to.
	 *
	 * @param int $event_id The event ID to fetch.
	 * @return array
	 */
	public static function get_users_invited_count( $event_id ) {
		global $wpdb;

		return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$wpdb->di_event_attending} WHERE event_id = %d AND status = 'invite'", $event_id ) );
	}

	/**
	 * Bulk check attending users based on an array of events
	 *
	 * @param array $event_ids Array of events.
	 * @param int   $user_id   User ID.
	 *
	 * @return array
	 */
	public static function bulk_check_attending_users( $event_ids, $user_id = false ) {
		global $wpdb;

		if ( empty( $user_id ) ) {
			$user_id = bp_loggedin_user_id();
		}

		if ( empty( $user_id ) ) {
			return false;
		}

		$event_ids = implode( ',', wp_parse_id_list( (array) $event_ids ) );

		return $wpdb->get_results( $wpdb->prepare( "SELECT event_id, id, status FROM {$wpdb->di_event_attending} WHERE user_id = %d AND event_id IN ( %s )", $user_id, $event_ids ) );
	}

	/**
	 * Deletes all attendings for a specific user
	 *
	 * @param int $user_id The user ID.
	 */
	public static function delete_all_for_user( $user_id ) {
		global $wpdb;

		$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->di_event_attending} WHERE user_id = %d", $user_id ) );
	}
}
