<?php
/**
 * Event Attend Rest Endpoint
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components\Rest;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Event Attend Class
 */
class Event_Attend_Controller extends \WP_REST_Controller {

	/**
	 * Parent controller
	 *
	 * @var object
	 */
	private $parent_controller;

	/**
	 * Parent base
	 *
	 * @var string
	 */
	private $parent_base;

	/**
	 * Constructor
	 */
	public function __construct() {
		$post_type = nisje_get_setting( 'post_type_name', 'event' );

		$this->parent_controller = new \WP_REST_Posts_Controller( $post_type );
		$this->parent_base       = $this->parent_controller->rest_base;
		$this->rest_base         = 'attend';
		$this->hook_base         = $this->parent_base . '_attend';
	}

	/**
	 * Register Routes
	 */
	public function register_routes() {
		register_rest_route( $this->parent_controller->namespace, '/' . $this->parent_base . '/' . $this->rest_base, [
			[
				'methods'             => \WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'attend' ],
				'permission_callback' => [ $this, 'attend_permissions_check' ],
				'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
			],
			'schema' => [ $this, 'get_public_item_schema' ],
		] );
	}

	/**
	 * Check if a given request has access to accept invite.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function attend_permissions_check( $request ) {
		$auth = nisje_validate_rest_authentication( $this->hook_base, 'attend' );
		if ( is_wp_error( $auth ) ) {
			return $auth;
		}

		$user_id = bp_loggedin_user_id();
		if ( ! empty( $request['user_id'] ) ) {
			$user_id = (int) $request['user_id'];
		}

		$user_check = nisje_validate_current_user( $user_id );
		if ( is_wp_error( $user_check ) ) {
			return $user_check;
		}

		$excluded = nisje_validate_rest_excluded( $user_id );
		if ( is_wp_error( $excluded ) ) {
			return $excluded;
		}

		$post_types = [
			nisje_get_setting( 'post_type_name', 'event' ),
			nisje_get_setting( 'post_type_name_group', 'event' ),
		];

		$event_id = (int) $request['event_id'];
		$event    = get_post( $event_id );
		if ( ! $event_id || ! $event || ! in_array( $event->post_type, $post_types, true ) ) {
			return new \WP_Error( 'nisje_rest_event_not_found', esc_html__( 'The event was not found.', 'nisje' ), [
				'status' => 404,
			] );
		}

		$field_key = nisje_get_setting( 'field_key', 'event' );
		$group_id  = (int) get_post_meta( $event->ID, $field_key . '_user_group', true );

		if ( $group_id ) {
			$group = nisje_get_group( $group_id );
			if ( is_wp_error( $group ) ) {
				return $group;
			}

			if ( 'public' !== $group->status && ! groups_is_user_member( $user_id, $group->id ) ) {
				return new \WP_Error( 'nisje_rest_event_no_access', esc_html__( 'Sorry, you are not allowed to attend to events from this group.', 'nisje' ), [
					'status' => rest_authorization_required_code(),
				] );
			}
		}

		return true;
	}

	/**
	 * Attend
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function attend( $request ) {
		$user_id = bp_loggedin_user_id();
		if ( ! empty( $request['user_id'] ) ) {
			$user_id = $request['user_id'];
		}

		$status = $request['status'];

		$event_id = (int) $request['event_id'];

		if ( ! nisje_event_connect_user( $event_id, $user_id, $status ) ) {
			return new \WP_Error( 'nisje_rest_failed_attending', esc_html__( 'The user could not attend to event. Please contact the administrator with error code 2501.', 'nisje' ), [
				'status' => 500,
			] );
		}

		$obj = new \stdClass();

		$obj->event_id = $event_id;
		$obj->user_id  = $user_id;
		$obj->status   = $status;

		$schema = $this->get_item_schema();

		/**
		 * Fires after users are invited.
		 */
		do_action( "nisje_rest_after_{$this->hook_base}", $obj, $request );

		$request->set_param( 'context', 'view' );
		$response = $this->prepare_item_for_response( $obj, $request );
		$response = rest_ensure_response( $response );
		$response->set_status( 201 );
		$response->header( 'Location', rest_url( sprintf( '%s/%s', $this->namespace, $this->rest_base ) ) );

		return $response;
	}

	/**
	 * Prepares the return value
	 *
	 * @param stdClass        $item    Group data.
	 * @param WP_REST_Request $request Request.
	 * @param boolean         $is_raw  Optional, not used. Defaults to false.
	 * @return WP_REST_Response
	 */
	public function prepare_item_for_response( $item, $request, $is_raw = false ) {
		$schema = $this->get_item_schema();

		$data = [];

		if ( ! empty( $schema['properties']['event_id'] ) ) {
			$data['event_id'] = (int) $item->event_id;
		}

		if ( ! empty( $schema['properties']['user_id'] ) ) {
			$data['user_id'] = (int) $item->user_id;
		}

		if ( ! empty( $schema['properties']['status'] ) ) {
			$data['status'] = $item->status;
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );

		$data     = $this->filter_response_by_context( $data, $context );
		$response = rest_ensure_response( $data );

		/**
		 * Filter value returned from the API.
		 */
		return apply_filters( "nisje_rest_prepare_{$this->hook_base}_value", $response, $item, $request );
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = [
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => "{$this->parent_base}-{$this->rest_base}",
			'type'       => 'object',
			'properties' => [
				'event_id' => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'A unique alphanumeric ID of the event to attend.', 'nisje' ),
					'type'        => 'integer',
					'required'    => true,
				],
				'user_id'  => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'A unique alphanumeric ID of the user to attend.', 'nisje' ),
					'type'        => 'integer',
				],
				'status'   => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'Status for the user.', 'nisje' ),
					'type'        => 'string',
					'enum'        => [ 'attend', 'not', 'maybe' ],
					'required'    => true,
				],
			],
		];

		return $this->add_additional_fields_schema( $schema );
	}
}
