<?php
/**
 * Event Helper Functions
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Attending an event
 *
 * @param int    $event_id   Event ID.
 * @param int    $user_id    User ID.
 * @param string $status     Status.
 * @param int    $inviter_id Inviter ID.
 *
 * @return bool
 */
function nisje_event_connect_user( $event_id, $user_id, $status, $inviter_id = 0 ) {
	if ( ! in_array( $status, [ 'attend', 'not', 'maybe', 'invite' ], true ) ) {
		return false;
	}

	$connect = new Nisje_Event_Attend( $event_id, $user_id, $status, $inviter_id );

	if ( ! $connect->save() ) {
		return false;
	}

	do_action( 'nisje_event_connected', $event_id, $user_id, $status, $inviter_id );

	return true;
}

/**
 * Remove connection.
 *
 * @param int    $event_id Event ID.
 * @param int    $user_id  User ID.
 * @param string $status   Status.
 *
 * @return bool
 */
function nisje_event_remove_connection( $event_id, $user_id, $status ) {
	if ( ! in_array( $status, [ 'attend', 'not', 'maybe', 'invite' ], true ) ) {
		return false;
	}

	$connect = new Nisje_Event_Attend( $event_id, $user_id, 'delete' );

	if ( ! $connect->delete() ) {
		return false;
	}

	do_action( 'nisje_event_removed_connection', $event_id, $user_id, $status );

	return true;
}

/**
 * Attending an event
 *
 * @param int $event_id Event ID.
 * @param int $user_id  User ID.
 *
 * @return array
 */
function nisje_event_attending( $event_id, $user_id ) {
	return nisje_event_connect_user( $event_id, $user_id, 'attend' );
}

/**
 * Not attending an event
 *
 * @param int $event_id Event ID.
 * @param int $user_id  User ID.
 *
 * @return array
 */
function nisje_event_not_attending( $event_id, $user_id ) {
	return nisje_event_connect_user( $event_id, $user_id, 'not' );
}

/**
 * Maybe attending
 *
 * @param int $event_id Event ID.
 * @param int $user_id  User ID.
 *
 * @return array
 */
function nisje_event_maybe_attending( $event_id, $user_id ) {
	return nisje_event_connect_user( $event_id, $user_id, 'maybe' );
}

/**
 * Invite to event
 *
 * @param int $event_id   Event ID.
 * @param int $user_id    User ID.
 * @param int $inviter_id Inviter ID.
 * @return array
 */
function nisje_event_invite( $event_id, $user_id, $inviter_id = 0 ) {
	return nisje_event_connect_user( $event_id, $user_id, 'invite', $inviter_id );
}

/**
 * Check if a user is attending
 *
 * @param int $event_id Event ID.
 * @param int $user_id    User ID.
 * @return array
 */
function nisje_event_is_user_attending( $event_id, $user_id ) {
	$attending = new Nisje_Event_Attend( $event_id, $user_id, 'attend' );

	return apply_filters( 'nisje_event_is_attending', $attending );
}

/**
 * Users attending to event
 *
 * @param int $event_id Event ID.
 * @return array
 */
function nisje_event_get_users_attending( $event_id ) {
	return apply_filters( 'nisje_event_get_users_attending', Nisje_Event_Attend::get_users_attending( $event_id ) );
}

/**
 * Users not attending to event
 *
 * @param int $event_id Event ID.
 * @return array
 */
function nisje_event_get_users_not_attending( $event_id ) {
	return apply_filters( 'nisje_event_get_users_not_attending', Nisje_Event_Attend::get_users_not_attending( $event_id ) );
}

/**
 * Users maybe attending to event
 *
 * @param int $event_id Event ID.
 * @return array
 */
function nisje_event_get_users_maybe_attending( $event_id ) {
	return apply_filters( 'nisje_event_get_users_maybe_attending', Nisje_Event_Attend::get_users_maybe_attending( $event_id ) );
}

/**
 * Users maybe attending to event
 *
 * @param int $event_id Event ID.
 * @return array
 */
function nisje_event_get_users_invited( $event_id ) {
	return apply_filters( 'nisje_event_get_users_invited', Nisje_Event_Attend::get_users_invited( $event_id ) );
}
