<?php
/**
 * Extend Event Endpoints with Meta Fields
 *
 * @package Nisje
 */

$field_key     = nisje_get_setting( 'field_key', 'event' );
$extend_object = [
	nisje_get_setting( 'post_type_rest_endpoint', 'event' ),
	nisje_get_setting( 'post_type_rest_endpoint_group', 'event' ),
	'feed',
];

// Add start date.
register_rest_field( $extend_object, 'start_date', [
	'get_callback'    => function( $object ) use ( $field_key ) {
		return get_post_meta( $object['id'], $field_key . '_start_date', true );
	},
	'update_callback' => function( $value, $object ) use ( $field_key ) {
		update_post_meta( $object->ID, $field_key . '_start_date', $value );

		return $value;
	},
	'schema'          => [
		'context'     => [ 'view', 'edit' ],
		'description' => esc_html__( 'Start date', 'nisje' ),
		'type'        => 'string',
		'format'      => 'date-time',
		'required'    => true,
	],
] );

// Add end date.
register_rest_field( $extend_object, 'end_date', [
	'get_callback'    => function( $object ) use ( $field_key ) {
		return get_post_meta( $object['id'], $field_key . '_end_date', true );
	},
	'update_callback' => function( $value, $object ) use ( $field_key ) {
		update_post_meta( $object->ID, $field_key . '_end_date', $value );

		return $value;
	},
	'schema'          => [
		'context'     => [ 'view', 'edit' ],
		'description' => esc_html__( 'End date', 'nisje' ),
		'type'        => 'string',
		'format'      => 'date-time',
	],
] );

// Add location.
register_rest_field( $extend_object, 'location', [
	'get_callback'    => function( $object ) use ( $field_key ) {
		return esc_html( get_post_meta( $object['id'], $field_key . '_location_text', true ) );
	},
	'update_callback' => function( $value, $object ) use ( $field_key ) {
		$value = esc_html( $value );
		update_post_meta( $object->ID, $field_key . '_location_text', $value );

		return $value;
	},
	'schema'          => [
		'context'     => [ 'view', 'edit' ],
		'description' => esc_html__( 'Location', 'nisje' ),
		'type'        => 'string',
	],
] );

// Add Attending.
register_rest_field( $extend_object, 'responses', [
	'get_callback' => function( $object ) {
		$not_attending = nisje_event_get_users_not_attending( $object['id'] );
		$attending     = nisje_event_get_users_attending( $object['id'] );
		$maybe         = nisje_event_get_users_maybe_attending( $object['id'] );

		return [
			'not'    => $not_attending['users'],
			'attend' => $attending['users'],
			'maybe'  => $maybe['users'],
		];
	},
	'schema'       => [
		'context'     => [ 'view' ],
		'description' => esc_html__( 'Users ids of users that is attending to this event.', 'nisje' ),
		'type'        => 'array',
		'items'       => [
			'type' => 'array',
		],
		'readonly'    => true,
	],
] );

// Add Maybe Attending.
register_rest_field( $extend_object, 'invited', [
	'get_callback' => function( $object ) {
		return nisje_event_get_users_invited( $object['id'] );
	},
	'schema'       => [
		'description' => esc_html__( 'Users ids of users that is invited to this event.', 'nisje' ),
		'type'        => 'array',
		'items'       => [
			'type' => 'integer',
		],
		'readonly'    => true,
	],
] );

// Add Maybe Attending.
register_rest_field( $extend_object, 'featuredImage', [
	'get_callback' => function( $object ) {
		if ( has_post_thumbnail( $object['id'] ) ) {
			$attachment_id = get_post_thumbnail_id( $object['id'] );
			$image         = wp_get_attachment_image_src( $attachment_id, 'medium_large', false );

			if ( $image ) {
				list( $src, $width, $height ) = $image;

				$attr = [
					'src' => $src,
					'alt' => trim( strip_tags( get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) ) ), // phpcs:ignore
				];

				// Generate 'srcset' and 'sizes' if not already present.
				if ( empty( $attr['srcSet'] ) ) {
					$image_meta = wp_get_attachment_metadata( $attachment_id );

					if ( is_array( $image_meta ) ) {
						$size_array = [ absint( $width ), absint( $height ) ];
						$srcset     = wp_calculate_image_srcset( $size_array, $src, $image_meta, $attachment_id );
						$sizes      = wp_calculate_image_sizes( $size_array, $src, $image_meta, $attachment_id );

						if ( $srcset && ( $sizes || ! empty( $attr['sizes'] ) ) ) {
							$attr['srcSet'] = $srcset;

							if ( empty( $attr['sizes'] ) ) {
								$attr['sizes'] = $sizes;
							}
						}
					}
				}

				$attr = array_map( 'esc_attr', $attr );

				return $attr;
			} else {
				return [];
			}
		}

		return [];
	},
	'schema'       => [
		'context'     => [ 'view', 'edit' ],
		'description' => esc_html__( 'Image data', 'nisje' ),
		'type'        => 'array',
		'items'       => [
			'type' => 'string',
		],
		'readonly'    => true,
	],
] );
