<?php
/**
 * Event Invite Rest Endpoint
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components\Rest;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Event Invite Class
 */
class Event_Invite_Controller extends \WP_REST_Controller {

	/**
	 * Parent controller
	 *
	 * @var object
	 */
	private $parent_controller;

	/**
	 * Parent base
	 *
	 * @var string
	 */
	private $parent_base;

	/**
	 * Constructor
	 */
	public function __construct() {
		$post_type = nisje_get_setting( 'post_type_name', 'event' );

		$this->parent_controller = new \WP_REST_Posts_Controller( $post_type );
		$this->parent_base       = $this->parent_controller->rest_base;
		$this->rest_base         = 'invite';
		$this->hook_base         = $this->parent_base . '_invite';
	}

	/**
	 * Register Routes.
	 */
	public function register_routes() {
		register_rest_route( $this->parent_controller->namespace, '/' . $this->parent_base . '/' . $this->rest_base, [
			[
				'methods'             => \WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'invite' ],
				'permission_callback' => [ $this, 'invite_permissions_check' ],
				'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
			],
			'schema' => [ $this, 'get_public_item_schema' ],
		] );
	}

	/**
	 * Check if a given request has access to accept invite.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function invite_permissions_check( $request ) {
		$auth = nisje_validate_rest_authentication( $this->hook_base, 'invite' );
		if ( is_wp_error( $auth ) ) {
			return $auth;
		}

		$user_ids   = $request['user_ids'];
		$inviter_id = bp_loggedin_user_id();
		if ( ! empty( $request['inviter_id'] ) ) {
			$inviter_id = (int) $request['inviter_id'];
		}

		if ( empty( $user_ids ) ) {
			return new \WP_Error( 'nisje_rest_missing_users', esc_html__( 'Missing user ids.', 'nisje' ), [
				'status' => 404,
			] );
		}

		$user_check = nisje_validate_current_user( $inviter_id );
		if ( is_wp_error( $user_check ) ) {
			return $user_check;
		}

		$post_types = [
			nisje_get_setting( 'post_type_name', 'event' ),
			nisje_get_setting( 'post_type_name_group', 'event' ),
		];

		$event_id = (int) $request['event_id'];
		$event    = get_post( $event_id );

		if ( ! $event_id || ! $event instanceof \WP_Post || ! in_array( $event->post_type, $post_types, true ) ) {
			return new \WP_Error( 'nisje_rest_event_not_found', esc_html__( 'The event was not found.', 'nisje' ), [
				'status' => rest_authorization_required_code(),
			] );
		}

		$group_id = (int) get_post_meta( $event->ID, nisje_get_setting( 'group_connection_key', 'event' ), true );
		if ( $group_id ) {
			$group = nisje_get_group( $group_id );
			if ( is_wp_error( $group ) ) {
				return $group;
			}

			if ( 'public' !== $group->status && ! groups_is_user_member( $inviter_id, $group->id ) ) {
				return new \WP_Error( 'nisje_rest_event_no_access', esc_html__( 'Sorry, you are not allowed to invite users to events from this group.', 'nisje' ), [
					'status' => rest_authorization_required_code(),
				] );
			}
		}

		return true;
	}

	/**
	 * Invite
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function invite( $request ) {
		$inviter_id = bp_loggedin_user_id();
		if ( ! empty( $request['inviter_id'] ) ) {
			$inviter_id = (int) $request['inviter_id'];
		}

		$event_id = (int) $request['event_id'];
		$event    = get_post( $event_id );

		$user_ids = $request['user_ids'];

		$invite_obj = new \stdClass();

		$invite_obj->inviter_id      = $inviter_id;
		$invite_obj->event_id        = $event_id;
		$invite_obj->invited_ids     = [];
		$invite_obj->not_invited_ids = [];

		if ( ! empty( $user_ids ) ) {
			$group_id = (int) get_post_meta( $event->ID, nisje_get_setting( 'group_connection_key', 'event' ), true );
			if ( $group_id ) {
				$group = nisje_get_group( $group_id );
				if ( is_wp_error( $group ) ) {
					return $group;
				}
			}

			foreach ( $user_ids as $user_id ) {
				$parse = true;

				if ( $group_id && 'public' !== $group->status && ! groups_is_user_member( $user_id, $group_id ) ) {
					$parse = false;

					$invite_obj->not_invited_ids[] = $user_id;
				}

				if ( $parse ) {
					if ( nisje_event_connect_user( $event_id, $user_id, 'invite', $inviter_id ) ) {
						$invite_obj->invited_ids[] = $user_id;

						do_action( 'nisje_event_invited_user', $user_id, $event_id, $inviter_id );
					}
				}
			}
		}

		$schema = $this->get_item_schema();

		/**
		 * Fires after users are invited.
		 */
		do_action( "nisje_rest_after_{$this->hook_base}", $invite_obj, $request );

		$request->set_param( 'context', 'view' );
		$response = $this->prepare_item_for_response( $invite_obj, $request );
		$response = rest_ensure_response( $response );
		$response->set_status( 201 );
		$response->header( 'Location', rest_url( sprintf( '%s/%s', $this->namespace, $this->rest_base ) ) );

		return $response;
	}

	/**
	 * Prepares the return value
	 *
	 * @param stdClass        $item    Group data.
	 * @param WP_REST_Request $request Request.
	 * @param boolean         $is_raw  Optional, not used. Defaults to false.
	 * @return WP_REST_Response
	 */
	public function prepare_item_for_response( $item, $request, $is_raw = false ) {
		$schema = $this->get_item_schema();

		$data = [];

		if ( ! empty( $schema['properties']['event_id'] ) ) {
			$data['event_id'] = (int) $item->event_id;
		}

		if ( ! empty( $schema['properties']['inviter_id'] ) ) {
			$data['inviter_id'] = (int) $item->inviter_id;
		}

		if ( ! empty( $schema['properties']['invited_ids'] ) ) {
			$data['invited_ids'] = $item->invited_ids;
		}

		if ( ! empty( $schema['properties']['not_invited_ids'] ) ) {
			$data['not_invited_ids'] = $item->not_invited_ids;
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );

		$data     = $this->filter_response_by_context( $data, $context );
		$response = rest_ensure_response( $data );

		/**
		 * Filter value returned from the API.
		 */
		return apply_filters( "nisje_rest_prepare_{$this->hook_base}_value", $response, $item, $request );
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = [
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => "{$this->hook_base}",
			'type'       => 'object',
			'properties' => [
				'event_id'        => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'A unique alphanumeric ID of the event to attend.', 'nisje' ),
					'type'        => 'integer',
					'required'    => true,
				],
				'inviter_id'      => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'The id of inviter. Fallback to logged in user if not set.', 'nisje' ),
					'type'        => 'integer',
				],
				'user_ids'        => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'Array of users to invite.', 'nisje' ),
					'type'        => 'array',
					'items'       => [
						'type' => 'integer',
					],
					'required'    => true,
				],
				'invited_ids'     => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'Invited users', 'nisje' ),
					'type'        => 'array',
					'items'       => [
						'type' => 'integer',
					],
					'readonly'    => true,
				],
				'not_invited_ids' => [
					'context'     => [ 'view' ],
					'description' => esc_html__( 'Users not invited', 'nisje' ),
					'type'        => 'array',
					'items'       => [
						'type' => 'integer',
					],
					'readonly'    => true,
				],
			],
		];

		return $this->add_additional_fields_schema( $schema );
	}
}
