<?php
/**
 * Setup Event Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Event Component Class
 */
class Event extends Component {
	/**
	 * Key
	 *
	 * @var string
	 */
	private $key;

	/**
	 * Constructor
	 */
	public function __construct() {
		$path = trailingslashit( dirname( __FILE__ ) );

		parent::start(
			'event',
			__( 'Event', 'nisje' ),
			$path
		);
	}

	/**
	 * Setup Globals
	 */
	public function setup_globals() {
		$this->field_key = 'dekode_event';
		$this->key       = 'di_event';

		$this->settings = [
			// Public.
			'post_type_name'                => 'dekode-event',
			'post_type_rest_endpoint'       => 'dekode-event',
			'capability_key'                => 'di_event', // nisje_event_cap.
			// Group.
			'post_type_name_group'          => 'nisje-event-group',
			'post_type_rest_endpoint_group' => 'nisje-event-group',
			'capability_key_group'          => 'nisje_event_cap_group',
			// General.
			'field_key'                     => $this->field_key,
			'key'                           => $this->key,
			'group_connection_key'          => '_nisje_group_connection',
		];
	}

	/**
	 * Include required files
	 */
	public function includes() {
		$this->include_file( 'helper-functions.php' );
		$this->include_file( 'classes/class-nisje-event-attend.php' );
	}

	/**
	 * Set up the default hooks and actions.
	 */
	public function setup_filters() {
		add_filter( 'nisje_feed_post_types', [ $this, 'add_feed_post_types' ] );
		add_filter( 'nisje_rest_shared_fields_group_post_types', [ $this, 'add_shared_rest_group_fields' ] );
		add_filter( 'nisje_taxonomy_map_group_types_object_types', [ $this, 'map_group_type' ] );
		add_filter( 'nisje_event_type_object_types', [ $this, 'map_group_type' ] );
		add_filter( 'nisje_event_region_object_types', [ $this, 'map_group_type' ] );
		add_filter( 'nisje_group_features', [ $this, 'add_group_feature' ] );
		add_filter( 'nisje_frontend_options', [ $this, 'add_frontend_option' ] );

		// Filter query by date from meta.
		add_filter( "rest_{$this->settings['post_type_name_group']}_query", [ $this, 'filter_rest_query_date_args' ], 10, 2 );
		add_filter( "rest_{$this->settings['post_type_name']}_query", [ $this, 'filter_rest_query_date_args' ], 10, 2 );
		add_filter( "rest_{$this->settings['post_type_name']}_archive_query", [ $this, 'filter_rest_query_date_args' ], 10, 2 );
		add_filter( "rest_{$this->settings['post_type_name']}_archive_query", [ $this, 'populate_group_type_to_archive' ], 10, 2 );

		add_filter( "nisje_group_post_validate_against_admin_{$this->settings['post_type_name_group']}", '__return_false' );

		// Clean up Attend data.
		add_action( 'wpmu_delete_user', [ $this, 'remove_attend_data' ] );
		add_action( 'delete_user', [ $this, 'remove_attend_data' ] );
		add_action( 'make_spam_user', [ $this, 'remove_attend_data' ] );

		if ( is_admin() ) {
			add_action( 'enqueue_block_editor_assets', [ $this, 'editor_assets' ], 20 );
		}
	}

	/**
	 * Editor mods
	 */
	public function editor_assets() {
		wp_enqueue_script( 'nisje-plugin-events-admin' );
		wp_enqueue_style( 'nisje-plugin-events-admin' );
	}

	/**
	 * Register content types
	 */
	public function register_content_types() {
		$this->include_file( 'post-types/post-type-event.php' );

		if ( bp_is_active( 'groups' ) ) {
			$this->include_file( 'post-types/post-type-event-group.php' );
		}
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		$this->include_file( 'rest-handlers/extend-event-shared.php' );

		$this->include_file( 'rest-handlers/class-event-attend-controller.php' );
		$controller = new \Dekode\Nisje\Components\Rest\Event_Attend_Controller();
		$controller->register_routes();

		$this->include_file( 'rest-handlers/class-event-invite-controller.php' );
		$controller = new \Dekode\Nisje\Components\Rest\Event_Invite_Controller();
		$controller->register_routes();

		$controller = new \Dekode\Nisje\Components\Rest\Base_Archive_Post_Controller( $this->settings['post_type_name'] );
		$controller->register_routes();

		if ( bp_is_active( 'groups' ) ) {
			$this->include_file( 'rest-handlers/extend-groups.php' );
		}
	}

	/**
	 * Add event settings to frontend options
	 *
	 * @param array $options Options.
	 */
	public function add_frontend_option( $options ) {
		if ( \is_user_logged_in() ) {
			$user_id = \get_current_user_id();
			$user    = \get_userdata( $user_id );

			$options['userCapabilities']['create']['event'] = $user->allcaps['publish_di_events'] ?? false;
		}

		return $options;
	}

	/**
	 * Add group features
	 *
	 * @param array $features Features.
	 */
	public function add_group_feature( $features ) {
		$features['enable_event'] = [
			'label'       => __( 'Events', 'nisje' ),
			'description' => __( 'Enable group events', 'nisje' ),
		];

		return $features;
	}

	/**
	 * Add feed post types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_feed_post_types( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_name_group', $this->settings_key );
		$post_types[] = nisje_get_setting( 'post_type_name', $this->settings_key );

		return $post_types;
	}

	/**
	 * Extend rest group types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_shared_rest_group_fields( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_rest_endpoint_group', $this->settings_key );
		$post_types[] = nisje_get_setting( 'post_type_name', $this->settings_key );

		return $post_types;
	}

	/**
	 * Map group types object types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function map_group_type( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_name_group', $this->settings_key );

		return $post_types;
	}

	/**
	 * Filters the query arguments for a request.
	 *
	 * @param array           $args    Key value array of query var to query value.
	 * @param WP_REST_Request $request The request used.
	 */
	public function filter_rest_query_date_args( $args, $request ) {
		if ( ! isset( $request['start_date'] ) ) {
			$args['start_date'] = current_time( 'mysql' );
		}

		if ( ! isset( $request['end_date'] ) ) {
			$args['end_date'] = $args['start_date'];
		}

		$args['orderby'] = [
			'regular_clause' => 'ASC',
		];

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
		$args['meta_query'] = [
			'relation'       => 'OR',
			'regular_clause' => [
				'key'     => $this->field_key . '_start_date',
				'value'   => $args['start_date'],
				'type'    => 'DATETIME',
				'compare' => '>',
			],
			'end_clause'     => [
				'key'     => $this->field_key . '_end_date',
				'value'   => $args['end_date'],
				'type'    => 'DATETIME',
				'compare' => '>',
			],
		];
		// phpcs:enable

		return $args;
	}

	/**
	 * Filters the query arguments for a request.
	 *
	 * @param array           $args    Key value array of query var to query value.
	 * @param WP_REST_Request $request The request used.
	 */
	public function populate_group_type_to_archive( $args, $request ) {
		$args['post_type'] = [
			$this->settings['post_type_name'],
			$this->settings['post_type_name_group'],
		];

		return $args;
	}

	/**
	 * Removes follow relationships for all users from a user who is deleted or spammed
	 *
	 * @param int $user_id User id.
	 */
	public function remove_attend_data( $user_id ) {
		\Nisje_Event_Attend::delete_all_for_user( $user_id );
	}
}
