<?php
/**
 * Layouts registry class
 *
 * @package Nisje
 */

/**
 * Class collecting layouts
 */
final class FPL_Layout_Registry {
	/**
	 * Registered layouts
	 *
	 * @var FPL_Layout[]
	 */
	private $registered_layouts = [];

	/**
	 * Instance
	 *
	 * @var FPL_Layout_Registry|null
	 */
	private static $instance = null;

	/**
	 * Instance.
	 *
	 * @return FPL_Layout_Registry The main instance.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Registers a layout.
	 *
	 * @param string $name Layout name.
	 * @param array  $args Arguments.
	 * @return FPL_Layout|false The registered layout on success, or false on failure.
	 */
	public function register( $name, $args = [] ) {
		if ( ! is_string( $name ) ) {
			_doing_it_wrong( __METHOD__, esc_html__( 'Name must be strings.', 'nisje-fpl' ), '0.1.0' );
			return false;
		}

		if ( preg_match( '/[A-Z]+/', $name ) ) {
			_doing_it_wrong( __METHOD__, esc_html__( 'Name must not contain uppercase characters.', 'nisje-fpl' ), '1.5.0' );
			return false;
		}

		if ( $this->is_registered( $name ) ) {
			/* translators: 1: layout name */
			$message = sprintf( __( '"%s" is already registered.', 'nisje-fpl' ), $name );
			_doing_it_wrong( __METHOD__, esc_html( $message ), '0.1.0' );
			return false;
		}

		$layout = new FPL_Layout( $name, $args );

		$this->registered_layouts[ $name ] = $layout;

		return $layout;
	}

	/**
	 * Unregister a layout.
	 *
	 * @param string $name Layout name.
	 * @return FPL_Layout|false The unregistered layout on success, or false on failure.
	 */
	public function unregister( $name ) {
		if ( ! $this->is_registered( $name ) ) {
			/* translators: 1: layout name */
			$message = sprintf( __( 'Layout "%s" is not registered.', 'nisje-fpl' ), $name );
			_doing_it_wrong( __METHOD__, esc_html( $message ), '0.1.0' );
			return false;
		}

		$unregistered_layout = $this->registered_layouts[ $name ];
		unset( $this->registered_layouts[ $name ] );

		return $unregistered_layout;
	}

	/**
	 * Retrieves a registered layout.
	 *
	 * @param string $name Layout name.
	 * @return FPL_Layout|null The registered layout or null
	 */
	public function get_registered( $name ) {
		if ( ! $this->is_registered( $name ) ) {
			return null;
		}

		return $this->registered_layouts[ $name ];
	}

	/**
	 * Retrieves all registered layout
	 *
	 * @return FPL_Layouts[] Associative array of `$layout_name => $layout` pairs.
	 */
	public function get_all_registered() {
		return $this->registered_layouts;
	}

	/**
	 * Checks if a layout is registered.
	 *
	 * @param string $name Layout name.
	 * @return bool True if the layout is registered, false otherwise.
	 */
	public function is_registered( $name ) {
		return isset( $this->registered_layouts[ $name ] );
	}
}
