<?php
/**
 * Helper functions.
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || exit;



/**
 * Registers a layout.
 *
 * @param string $name Name.
 * @param array  $args Arguments.
 *
 * @return FPL_Layout|false The registered layout on success, or false on failure.
 */
function nisje_fpl_register_layout( $name, $args = [] ) {
	return FPL_Layout_Registry::get_instance()->register( $name, $args );
}

/**
 * Registers a layout.
 *
 * @param string $name Name.
 *
 * @return FPL_Layout|false The unregistered layout on success, or false on failure.
 */
function nisje_fpl_unregister_layout( $name ) {
	return FPL_Layout_Registry::get_instance()->unregister( $name );
}

/**
 * Get all registered layouts.
 *
 * @return array Array of all registered layouts.
 */
function nisje_flp_get_all_registered_layouts() {
	return FPL_Layout_Registry::get_instance()->get_all_registered();
}

/**
 * Registered FPL_Layout
 *
 * @param string $name Name of the layout.
 * @return FPL_Layout|null Return layout if exists.
 */
function nisje_fpl_get_registered_layout( $name ) {
	return FPL_Layout_Registry::get_instance()->get_registered( $name );
}

/**
 * Get global front page layouts
 *
 * @param int $group_id Group ID.
 * @return array $layouts
 */
function nisje_fpl_build_layouts( $group_id = false ) {
	$layouts = [];

	$args = [
		'post_status' => 'publish',
	];

	if ( $group_id && is_numeric( $group_id ) ) {
		$args['post_type']  = nisje_get_setting( 'post_type_name_group', 'front-page-layouts' );
		$args['meta_query'] = [ //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			[
				'key'     => 'group_id',
				'value'   => (int) $group_id,
				'compare' => 'IN',
			],
		];
	} else {
		$args['post_type'] = nisje_get_setting( 'post_type_name', 'front-page-layouts' );
	}

	$loop = new \WP_Query( $args );

	if ( $loop->have_posts() ) {
		while ( $loop->have_posts() ) {
			$loop->the_post();

			$cards         = get_field( 'cards' );
			$not_logged_in = (int) get_field( 'not_logged_in' );

			$card_layouts = [];

			if ( $cards ) {
				foreach ( $cards as $key => $card ) {
					if ( isset( $card['acf_fc_layout'] ) ) {
						$retval = $card;

						$type   = $card['acf_fc_layout'];
						$layout = nisje_fpl_get_registered_layout( $type );

						if ( $layout && $layout->is_dynamic() ) {
							$retval['custom'] = $layout->render();
						}

						$username = isset( $card['username'] ) ? $card['username'] : '';
						if ( $type && 'tweet' === $type && $username ) {
							$data = nisje_get_tweets( $username );
							if ( $data && ! empty( $data ) ) {
								$retval['tweets'] = $data;
							}
						}

						$meta = apply_filters( 'nisje_fpl_layout_meta_' . $type, [], $card );
						if ( ! empty( $meta ) ) {
							$retval['meta'] = $meta;
						}

						$card_layouts[] = $retval;
					}
				}
			}

			$layouts[] = [
				'cards'         => $card_layouts,
				'not_logged_in' => $not_logged_in,
			];
		}
	}

	return $layouts;
}

/**
 * Fetch tweets
 *
 * @param string $username The username of the Twitter account.
 *
 * @return array $data The tweets from the account.
 */
function nisje_get_tweets( $username ) {
	// If cached tweets for this username, return them.
	$data = get_transient( 'tweets_' . $username );
	if ( $data ) {
		return $data;
	}

	$data            = [];
	$consumer_key    = get_field( 'twitter_consumer_key', 'option' );
	$consumer_secret = get_field( 'twitter_consumer_secret', 'option' );
	$access_token    = get_field( 'twitter_access_token', 'option' );
	$access_secret   = get_field( 'twitter_access_secret', 'option' );
	$api_function    = get_field( 'twitter_api_function', 'option' );

	if ( class_exists( '\Codebird\Codebird' ) && $username && $consumer_key && $consumer_secret && $access_token && $access_secret && $api_function ) {
		\Codebird\Codebird::setConsumerKey( $consumer_key, $consumer_secret );
		$cb = \Codebird\Codebird::getInstance();
		$cb->setToken( $access_token, $access_secret );
		$count = 1;
		$rt    = false;

		// https://developer.twitter.com/en/docs/tweets/timelines/api-reference/get-statuses-user_timeline.html.
		$params = [
			'screen_name' => $username,
			'count'       => $count,
			'include_rts' => $rt,
		];

		// Make the REST call.
		if ( ! empty( $api_function ) ) {
			$data = (array) $cb->$api_function( $params );
			if ( $data && ! empty( $data ) ) {
				set_transient( 'tweets_' . $username, $data, 3 * 60 );
			}
		}
	}

	return $data;
}
