<?php
/**
 * Setup Front Page Layouts Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

if ( ! class_exists( '\\Dekode\\Nisje\\Components\\Front_Page_Layouts' ) && class_exists( '\\Dekode\\Nisje\\Components\\Component' ) ) {

	/**
	 * Front Page Layouts Component Class
	 */
	class Front_Page_Layouts extends Component {

		/**
		 * Constructor
		 */
		public function __construct() {
			$path = trailingslashit( dirname( __FILE__ ) );
			$name = 'front-page-layouts';

			parent::start(
				$name,
				esc_html__( 'Front Page Layouts', 'nisje-fpl' ),
				$path,
				$name
			);
		}

		/**
		 * Global variables
		 */
		public function setup_globals() {
			$defaults = [
				'option_page_title'             => esc_html__( 'Front Page Layouts', 'nisje-fpl' ),
				'option_page_slug'              => 'front-page-layouts',
				// Public.
				'capability_key'                => 'nisje_front_page_layouts',
				'post_type_name'                => 'nisje-fpls',
				'post_type_rest_endpoint'       => 'nisje-front-page-layouts',
				// Group.
				'capability_key_group'          => 'nisje_front_page_layouts_group',
				'post_type_name_group'          => 'nisje-fpls-group',
				'post_type_rest_endpoint_group' => 'nisje-front-page-layouts-group',
				'cache_key'                     => 'nisje_fpl_cache',
			];

			$this->settings = wp_parse_args( $this->settings, $defaults );
		}

		/**
		 * Includes
		 */
		public function includes() {
			$this->include_file( 'classes/class-fpl-layout-registry.php' );
			$this->include_file( 'classes/class-fpl-layout.php' );

			$this->include_file( 'helper-functions.php' );
		}

		/**
		 * Setup filters
		 */
		public function setup_filters() {
			add_action( 'nisje_register_acf_fields', [ $this, 'register_field_groups' ] );

			if ( is_admin() ) {
				add_action( 'nisje_settings_add_option_page', [ $this, 'add_menu' ] );

				$post_type            = $this->settings['post_type_name'];
				$post_type_name_group = $this->settings['post_type_name_group'];

				add_action( "save_post_{$post_type}", [ $this, 'flush_transient_cache' ] );
				add_action( "save_post_{$post_type_name_group}", [ $this, 'flush_transient_cache' ] );
			}
		}

		/**
		 * Populate settings
		 */
		public function register_public_settings() {
			$this->public_settings = [
				'layouts' => nisje_fpl_build_layouts(),
			];
		}

		/**
		 * Register content types
		 */
		public function register_content_types() {
			$this->include_file( 'post-types/post-type-front-page.php' );
			$this->include_file( 'post-types/post-type-front-page-group.php' );
		}

		/**
		 * Include ACF field groups
		 */
		public function register_field_groups() {
			$this->include_file( 'layouts/custom.php' );
			$this->include_file( 'layouts/image-only.php' );
			$this->include_file( 'layouts/latest.php' );
			$this->include_file( 'layouts/links.php' );
			$this->include_file( 'layouts/single.php' );
			$this->include_file( 'layouts/tweet.php' );
			$this->include_file( 'layouts/user-content.php' );

			do_action( 'nisje_fpl_register_layouts' );

			$this->include_file( 'acf/field-groups/acf-field-group-front-page-layouts.php' );
			$this->include_file( 'acf/field-groups/acf-field-group-front-page-layout-group-connection.php' );
			$this->include_file( 'acf/field-groups/acf-field-group-options-front-page-layouts.php' );
			$this->include_file( 'acf/field-groups/acf-field-group-front-page-layout-not-logged-in.php' );
		}

		/**
		 * Setup the admin page.
		 */
		public function add_menu() {
			acf_add_options_sub_page(
				[
					'page_title'  => $this->settings['option_page_title'],
					'menu_title'  => $this->settings['option_page_title'],
					'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
					'menu_slug'   => $this->settings['option_page_slug'],
					'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
				]
			);
		}

		/**
		 * Register REST Endpoints
		 */
		public function register_rest_routes() {
			if ( bp_is_active( 'groups' ) ) {
				$this->include_file( 'rest-handlers/extend-groups.php' );
			}
		}

		/**
		 * Delete transient if there is a post saved in the backend.
		 *
		 * @param int $post_id Post ID.
		 */
		public function flush_transient_cache( $post_id ) {
			$cache_key = nisje_get_setting( 'cache_key', 'front-page-layouts' );
			if ( $cache_key ) {
				delete_transient( $cache_key );
			}
		}

		/**
		 * Get transient Cache.
		 *
		 * @param int $group_id Group ID.
		 * @return false|array
		 */
		public function get_transient_cache( $group_id = 0 ) {
			// Support group caching.
			$cache_key = nisje_get_setting( 'cache_key', 'front-page-layouts' );

			$retval = get_transient( $cache_key );
			if ( false === $retval ) {
				$retval = [
					'layouts' => nisje_fpl_build_layouts(),
				];

				set_transient( $cache_key, $retval, HOUR_IN_SECONDS );
			}

			return $retval;
		}
	}
}
