<?php
/**
 * Plugin Name: Nisje - GDPR
 * Description: Force users to accept privacy policy before entering Nisje
 * Author:      Dekode
 * Author URI:  https://dekode.no/
 * Version:     9.2.0
 * Text Domain: nisje
 * License:     GPLv2 or later (license.txt)
 *
 * @package Nisje
 */

declare( strict_types = 1 );
defined( 'ABSPATH' ) || die( 'Shame on you' );

define( 'NISJE_GDPR_VERSION', '9.2.0' );
define( 'NISJE_GDPR_PATH', \plugin_dir_path( __FILE__ ) );

/**
 * Add option page
 */
add_action( 'nisje_settings_add_option_page', function() {
	acf_add_options_sub_page( [
		'page_title'  => esc_html__( 'GDPR', 'nisje' ),
		'menu_title'  => esc_html__( 'GDPR', 'nisje' ),
		'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
		'menu_slug'   => 'nisje-gdpr',
		'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
	] );
} );

/**
 * Add fields
 */
add_action( 'nisje_settings_register_field_groups', function() {
	acf_add_local_field_group(
		[
			'key'                   => 'nisje_gdpr',
			'title'                 => esc_html__( 'GDPR Settings', 'nisje' ),
			'fields'                => [
				[
					'key'          => 'nisje_gdpr_info',
					'label'        => esc_html__( 'Info text above privacy terms', 'nisje' ),
					'name'         => 'nisje_gdpr_info',
					'type'         => 'wysiwyg',
					'role'         => '',
					'media_upload' => false,
					'toolbar'      => 'basic',
				],
			],
			'location'              => [
				[
					[
						'param'    => 'options_page',
						'operator' => '==',
						'value'    => 'nisje-gdpr',
					],
				],
			],
			'menu_order'            => 0,
			'position'              => 'normal',
			'style'                 => 'default',
			'label_placement'       => 'top',
			'instruction_placement' => 'label',
			'hide_on_screen'        => '',
			'active'                => 1,
			'description'           => '',
		]
	);
} );

/**
 * Include plugin components
 */
function nisje_gdpr() {
	if ( is_user_logged_in() && ! is_admin() ) {
		$meta_key     = 'gdpr_accepted_1';
		$user_id      = get_current_user_id();
		$has_accepted = get_user_meta( $user_id, $meta_key, true );
		$nonce_key    = 'nisje-gdpr';

		if ( empty( $has_accepted ) || ! $has_accepted ) {
			$nonce_check    = isset( $_REQUEST['_wpnonce'] ) && wp_verify_nonce( $_REQUEST['_wpnonce'], $nonce_key ); // phpcs:ignore
			if ( $nonce_check && isset( $_REQUEST['accept'] ) ) {
				update_user_meta( $user_id, $meta_key, true );
			} else {
				$policy_page_id = (int) get_option( 'wp_page_for_privacy_policy' );

				if ( ! empty( $policy_page_id ) && get_post_status( $policy_page_id ) === 'publish' && ! is_page( $policy_page_id ) ) {
					$policy_post  = get_post( $policy_page_id );
					$content      = apply_filters( 'the_content', $policy_post->post_content );
					$style_path   = get_stylesheet_directory_uri();
					$info_content = get_field( 'nisje_gdpr_info', 'option' );
					?>
					<html <?php language_attributes(); ?>>
						<head>
							<meta charset="<?php bloginfo( 'charset' ); ?>" />
							<meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no" />
							<title><?php echo get_bloginfo( 'name' ); // phpcs:ignore ?></title>
							<?php // phpcs:disable ?>
							<link rel="stylesheet" href="<?php echo $style_path; ?>/build/sanitize/style.css" type="text/css" media="all" />
							<link rel="stylesheet" href="<?php echo $style_path; ?>/build/settings/style.css" type="text/css" media="all" />
							<link rel="stylesheet" href="<?php echo $style_path; ?>/build/wordpress/privacy.css" type="text/css" media="all" />
							<link rel="stylesheet" href="<?php echo $style_path; ?>/build/components/style.css" type="text/css" media="all" />
							<link rel="stylesheet" href="<?php echo $style_path; ?>/assets/fonts/circular/style.css" type="text/css" media="all" />
							<?php // phpcs:enable ?>
						</head>
						<body>
							<div class="privacy-wrapper">
								<?php
								printf( '<h1>%s</h1>', esc_html( $policy_post->post_title ) );

								if ( ! empty( $info_content ) ) {
									printf( '<div class="privacy-info">%s</div>',
										wp_kses_post( apply_filters( 'the_content', get_field( 'nisje_gdpr_info', 'option' ) ) )
									);
								}

								echo wp_kses_post( $content );
								?>
							</div>
							<div class="privacy-accept">
								<div class="privacy-accept-content">
									<a href="<?php echo esc_url( wp_nonce_url( add_query_arg( 'accept', 'true' ), $nonce_key ) ); ?>" class="nisje-button nisje-button--button nisje-button--primary is-square-design"><?php esc_html_e( 'Accept', 'nisje' ); ?></a>
								</div>
							</div>
						</body>
					</html>
					<?php
					exit;
				}
			}
		}
	}
}
add_action( 'parse_request', 'nisje_gdpr' );
