<?php
/**
 * Signup Notifications
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

/**
 * Handles notifications which are sent here and there by email when a new user registers.
 *
 * Class Grieg_Signup_Notifications
 *
 * @package Dekode\Nisje\Components
 */
class Grieg_Signup_Notifications extends Component {
	/**
	 * Set up component
	 *
	 * Grieg_Signup_Notifications constructor.
	 */
	public function __construct() {
		$path = trailingslashit( __DIR__ );

		parent::start(
			'grieg-signup-notifications',
			esc_html__( 'Grieg Signup Notifications', 'nisje-grieg' ),
			$path
		);

	}

	/**
	 * Sets up filters and hooks.
	 */
	public function setup_filters() {
		add_action( 'bp_core_signup_user', [ $this, 'send_notification_to_admin' ], 10, 2 );
		add_action( 'bp_core_activated_user', [ $this, 'send_notification_to_user' ], 10, 1 );

		add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_group' ] );
		add_action( 'nisje_settings_add_option_page', [ $this, 'setup_settings_menu' ] );

	}

	/**
	 * Sets options for this component.
	 */
	public function setup_globals() {
		$this->settings = [
			'field_key_prefix'  => 'nisje_grieg_admin_notifications',
			'settings_menu_key' => 'nisje-settings-grieg-admin-notifications',
		];

	}


	/**
	 * Send an email to the specified admin email address when a user tries to register.
	 *
	 * @param  integer $user_id    The ID of the user who has registered.
	 * @param  object  $user_login The username of the user who has registered.
	 */
	public function send_notification_to_admin( $user_id, $user_login ) {
		$field_key_prefix = nisje_get_setting( 'field_key_prefix', $this->settings_key );

		$admin_email = get_field( $field_key_prefix . '_admin_email', 'option' );

		wp_mail( $admin_email, 'New registration', "A new user ($user_login) has registered on 135.grieg.com.\n\nClick this link to see and manage pending registration requests:\n\nhttps://135.grieg.com/wp/wp-admin/users.php?page=bp-signups", 'From: noreply@grieg.com' );

	}

	/**
	 * Sends an email to the user who tried to register when he/she has been approved.
	 *
	 * @param  integer $user_id The ID of the approved user.
	 */
	public function send_notification_to_user( $user_id ) {
		$field_key_prefix = nisje_get_setting( 'field_key_prefix', $this->settings_key );

		$user = get_user_by( 'id', $user_id );

		$subject = get_field( $field_key_prefix . '_user_approved_notification_subject', 'option' );
		$body    = get_field( $field_key_prefix . '_user_approved_notification_body', 'option' );

		$email = $user->data->user_email;
		wp_mail( $email, $subject, $body, [ 'Content-Type: text/html; charset=UTF-8' ] );

	}

	/**
	 * Setup ACF field groups.
	 */
	public function register_field_group() {

		$field_key_prefix = nisje_get_setting( 'field_key_prefix', $this->settings_key );
		acf_add_local_field_group(
			[
				'key'                   => $field_key_prefix . '_grieg_signup_notifications',
				'title'                 => esc_html__( 'Signup notifications', 'nisje-grieg' ),
				'fields'                => [
					[
						'key'          => $field_key_prefix . '_admin_email',
						'label'        => esc_html__( 'Admin email', 'nisje-grieg' ),
						'name'         => $field_key_prefix . '_admin_email',
						'type'         => 'email',
						'instructions' => esc_html__( 'When a new user registers, a notification will be sent to this email address.', 'nisje-grieg' ),
						'required'     => 1,
					],
					[
						'key'          => $field_key_prefix . '_user_approved_notification_subject',
						'label'        => esc_html__( 'Approval email subject', 'nisje-grieg' ),
						'name'         => $field_key_prefix . '_user_approved_notification_subject',
						'type'         => 'text',
						'instructions' => esc_html__( 'When a user has been approved, an email with this subject will be sent to him/her.', 'nisje-grieg' ),
						'required'     => 1,
					],
					[
						'key'          => $field_key_prefix . '_user_approved_notification_body',
						'label'        => esc_html__( 'Approval email body', 'nisje-grieg' ),
						'name'         => $field_key_prefix . '_user_approved_notification_body',
						'type'         => 'wysiwyg',
						'instructions' => esc_html__( 'When a user has been approved, an email with this body text will be sent to him/her.', 'nisje-grieg' ),
						'required'     => 1,
					],

				],
				'location'              => [
					[
						[
							'param'    => 'options_page',
							'operator' => '==',
							'value'    => nisje_get_setting( 'settings_menu_key', $this->settings_key ),
						],
					],
				],
				'menu_order'            => 0,
				'position'              => 'normal',
				'style'                 => 'default',
				'label_placement'       => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen'        => '',
				'active'                => 1,
				'description'           => '',
			]
		);
	}

	/**
	 * Setup the admin page.
	 */
	public function setup_settings_menu() {

		acf_add_options_sub_page(
			[
				'page_title'  => esc_html__( 'Signup Notifications', 'nisje-grieg' ),
				'menu_title'  => esc_html__( 'Signup Notifications', 'nisje-grieg' ),
				'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
				'menu_slug'   => nisje_get_setting( 'settings_menu_key', $this->settings_key ),
				'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
			]
		);
	}

}
