<?php
/**
 * Plugin Name: Nisje - JWT Auth
 * Description: Bring JWT Auth to Nisje
 * Author:      Dekode
 * Author URI:  https://dekode.no/en/
 * Version:     7.3.0
 * Text Domain: nisje-jwt
 * Domain Path: /languages/
 * License:     GPLv2 or later (license.txt)
 *
 * @package Nisje
 */

declare( strict_types = 1 );
defined( 'ABSPATH' ) || die( 'Shame on you' );

define( 'NISJE_JWT_VERSION', '7.3.0' );
define( 'NISJE_JWT_PATH', \plugin_dir_path( __FILE__ ) );

if ( file_exists( NISJE_JWT_PATH . 'vendor/autoload.php' ) ) {
	require NISJE_JWT_PATH . 'vendor/autoload.php';
}

require NISJE_JWT_PATH . 'rest-helpers.php';

/**
 * Load plugin textdomain.
 */
function nisje_jwt_load_textdomain() {
	load_plugin_textdomain( 'nisje-jwt', false, basename( __DIR__ ) . '/languages' );
}
add_action( 'plugins_loaded', 'nisje_jwt_load_textdomain' );

/**
 * Add API Routes
 */
function nisje_jwt_add_api_routes() {
	register_rest_route( 'jwt-auth/v1', 'token', [
		'methods'             => 'POST',
		'permission_callback' => '__return_true',
		'callback'            => 'nisje_jwt_generate_rest_token',
	] );

	register_rest_route( 'jwt-auth/v1', 'token/validate', [
		'methods'             => 'POST',
		'permission_callback' => '__return_true',
		'callback'            => 'nisje_jwt_validate_user',
	] );
}
add_action( 'rest_api_init', 'nisje_jwt_add_api_routes' );

/**
 * Add CORs suppot to the request.
 */
function nisje_add_cors_support() {
	$enable_cors = defined( 'JWT_AUTH_CORS_ENABLE' ) ? JWT_AUTH_CORS_ENABLE : false;
	if ( $enable_cors ) {
		$headers = apply_filters( 'jwt_auth_cors_allow_headers', 'Access-Control-Allow-Headers, Content-Type, Authorization' );
		header( sprintf( 'Access-Control-Allow-Headers: %s', $headers ) );
	}
}
add_filter( 'rest_pre_serve_request', 'nisje_add_cors_support' );

/**
 * Determine current user
 *
 * @param int|bool $user Current user.
 */
function nisje_jwt_determine_current_user( $user ) {
	$auth = nisje_jwt_check_auth_header();
	if ( ! $auth ) {
		return $user;
	}

	$request_uri   = filter_input( INPUT_SERVER, 'REQUEST_URI', FILTER_SANITIZE_STRING );
	$rest_api_slug = rest_get_url_prefix();

	$valid_api_uri = strpos( $request_uri, $rest_api_slug );
	if ( ! $valid_api_uri ) {
		return $user;
	}

	// Avoid revalidating.
	$validate_uri = strpos( $request_uri, 'token/validate' );
	if ( $validate_uri > 0 ) {
		return $user;
	}

	$token = nisje_jwt_validate_token( false );

	if ( is_wp_error( $token ) ) {
		if ( $token->get_error_code() !== 'jwt_auth_no_auth_header' ) {
			add_filter( 'rest_pre_dispatch', function( $request ) use ( $token ) {
				return $token;
			} );

			return false;
		} else {
			return $user;
		}
	}

	return $token->data->user->id;
}
add_filter( 'determine_current_user', 'nisje_jwt_determine_current_user', 10 );

/**
 * Add JWT token to theme
 */
add_action( 'init', function() {
	if ( is_user_logged_in() ) {
		add_filter( 'nisje_jwt_token', function( $token ) {
			return nisje_jwt_generate_token( wp_get_current_user() );
		} );
	}
} );

/**
 * Filter token data
 *
 * @param array   $token Token data.
 * @param WP_User $user  User.
 * @return array
 */
function nisje_add_token_data( $token, $user ) {
	if ( function_exists( 'nisje_get_setting' ) ) {
		$token['communityId'] = nisje_get_setting( 'community_id', 'general' );
		$token['userId']      = (int) $user->data->ID;
	}

	return $token;
}
add_filter( 'jwt_auth_token_before_sign', 'nisje_add_token_data', 10, 2 );
