<?php
/**
 * Setup Mandatory Groups Component
 *
 * @package Nisje
 */

declare( strict_types = 1 );
namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Mandatory Groups Component Class
 */
class Mandatory_Groups extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		$path  = trailingslashit( dirname( __FILE__ ) );
		$label = esc_html__( 'Mandatory Groups', 'nisje' );

		parent::start(
			'mandatory-groups',
			$label,
			$path
		);
	}

	/**
	 * Setup globals
	 */
	public function setup_globals() {
		$settings = get_option( $this->option_key );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			foreach ( $settings as $key => $value ) {
				$this->settings[ $key ] = $value;
			}
		}

		$defaults = [
			'name'              => $this->name,
			'option_key'        => $this->option_key,
			'field_key_prefix'  => 'nisje_mandatory_groups',
			'settings_menu_key' => 'nisje-settings-mandatory-groups',
		];

		$this->settings = array_merge( $defaults, (array) $this->settings );
	}

	/**
	 * Setup Filters
	 */
	public function setup_filters() {
		add_action( 'wp_login', [ $this, 'autojoin_users' ], 10, 2 );
		add_action( 'jwt_auth_token_before_dispatch', [ $this, 'autojoin_users_jwt' ], 10, 2 );
		add_filter( 'nisje_rest_groups_memberships_leave_permission', [ $this, 'reject_leaving_group' ], 10 );

		add_filter( 'nisje_frontend_options', [ $this, 'dom_options' ] );

		if ( is_admin() ) {
			add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_group' ] );
			add_action( 'nisje_settings_add_option_page', [ $this, 'setup_settings_menu' ] );
			add_action( 'nisje_settings_populate_options_from_acf', [ $this, 'populate_acf_settings' ] );
		}
	}

	/**
	 * Autojoin users to Mandatory Groups on login.
	 *
	 * @param string   $user_login User Login.
	 * @param \WP_User $user       User.
	 */
	public function autojoin_users( $user_login, $user ) {
		if ( $user instanceof \WP_User ) {
			$groups = nisje_get_setting( 'groups', 'mandatory_groups' );
			if ( $groups && ! empty( $groups ) ) {
				foreach ( $groups as $group ) {
					if ( ! groups_is_user_member( $user->ID, $group ) ) {
						groups_join_group( $group, $user->ID );
					}
				}
			}
		}
	}

	/**
	 * Autojoin users to Mandatory Groups on login.
	 *
	 * @param array    $data Data from JWT.
	 * @param \WP_User $user WP User object.
	 */
	public function autojoin_users_jwt( array $data, \WP_User $user ) : array {
		$groups = nisje_get_setting( 'groups', 'mandatory_groups' );
		if ( $groups && ! empty( $groups ) ) {
			foreach ( $groups as $group ) {
				if ( ! groups_is_user_member( $user->ID, $group ) ) {
					groups_join_group( $group, $user->ID );
				}
			}
		}

		return $data;
	}

	/**
	 * Do not allow users to reject groups
	 *
	 * @param array $data Data from JWT.
	 * @return array $data|\WP_Error Data from JWT.
	 */
	public function reject_leaving_group( array $data ) {
		$groups = nisje_get_setting( 'groups', 'mandatory_groups' );
		if ( in_array( $data['group']->id, $groups, true ) ) {
			return new \WP_Error( 'nisje_rest_mandatory_group', esc_html__( 'You are not allowed to leave mandatory groups.', 'nisje' ), [
				'status' => 400,
			] );
		}

		return $data;
	}

	/**
	 * Add DOM Options.
	 *
	 * @param array $options Options.
	 * @return array $options Options.
	 */
	public function dom_options( array $options ) : array {
		$options['mandatoryGroups'] = nisje_get_setting( 'groups', 'mandatory_groups' );

		return $options;
	}

	/**
	 * Setup the admin page.
	 */
	public function setup_settings_menu() {
		$label = esc_html__( 'Mandatory Groups', 'nisje' );

		acf_add_options_sub_page( [
			'page_title'  => $label,
			'menu_title'  => $label,
			'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
			'menu_slug'   => nisje_get_setting( 'settings_menu_key', $this->settings_key ),
			'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
		] );
	}

	/**
	 * Include acf field groups.
	 */
	public function register_field_group() {
		$this->include_file( 'acf/field-groups/acf-field-group-options-mandatory-groups.php' );
	}

	/**
	 * Populate ACF Options General
	 *
	 * @param \WP_Screen $screen Screen object.
	 */
	public function populate_acf_settings( \WP_Screen $screen ) {
		if ( false !== strpos( $screen->id, nisje_get_setting( 'settings_menu_key', $this->settings_key ) ) ) {
			$settings = [
				'groups' => [],
			];

			$field_key_prefix = nisje_get_setting( 'field_key_prefix', $this->settings_key );
			$groups_repeater  = get_field( $field_key_prefix . '_groups', 'option' );

			foreach ( (array) $groups_repeater as $group ) {
				if ( isset( $group['group'] ) ) {
					$settings['groups'][] = $group['group'];
				}
			}

			update_option( $this->option_key, $settings );
		}
	}
}
