<?php
/**
 * Map Library data.
 *
 * @package Nisje
 */

declare( strict_types=1 );

namespace Dekode\Nisje\Commands;

use WP_CLI;
use Exception;
use StdClass;
use WP_Post;

if ( class_exists( 'WP_CLI' ) ) {
	/**
	 * Map library meta.
	 */
	class NB_Library {
		/**
		 * Map meta
		 */
		public function map_meta() {
			$query_args = [
				'per_page'    => 10000,
				'show_hidden' => true,
				'fields'      => 'ids',
			];

			$groups = groups_get_groups( $query_args );
			if ( empty( $groups['groups'] ) ) {
				WP_CLI::error( 'No groups found.' );
			}

			$option_key_meta       = 'nb_library_meta';
			$taxonomy_type         = nisje_get_setting( 'taxonomy_type_name', 'nb_library' );
			$taxonomy_municipality = nisje_get_setting( 'taxonomy_municipality_name', 'nb_library' );
			$taxonomy_county       = nisje_get_setting( 'taxonomy_county_name', 'nb_library' );

			foreach ( $groups['groups'] as $group ) {
				echo 'Group ID: ' . absint( $group ) . "\n";

				$value = groups_get_groupmeta( $group, $option_key_meta, true );

				$data = json_decode( $value );
				if ( isset( $data->bibltype ) ) {
					wp_set_object_terms( $group, $data->bibltype, $taxonomy_type );
					echo 'Bibltype mapped: ' . esc_html( $group ) . ' - ' . esc_html( $data->bibltype ) . "\n";
				}

				if ( isset( $data->kommnr ) ) {
					wp_set_object_terms( $group, strval( $data->kommnr ), $taxonomy_municipality );
					echo 'Kommnr mapped: ' . esc_html( $group ) . ' - ' . esc_html( $data->kommnr ) . "\n";

					$county = substr( $data->kommnr, 0, 2 );
					wp_set_object_terms( $group, $county, $taxonomy_county );
					echo 'Fylkenr mapped: ' . esc_html( $group ) . ' - ' . esc_html( $county ) . "\n";
				}

				echo "\n";
			}
		}

		/**
		 * Create passwords
		 * This function is run on the command "wp nb-library create_passwords"
		 */
		public function create_passwords() {
			$query_args = [
				'per_page'    => 5000,
				'group_type'  => 'schoollibrary',
				'show_hidden' => true,
				'fields'      => 'ids',
			];

			$groups = groups_get_groups( $query_args );
			if ( empty( $groups['groups'] ) ) {
				WP_CLI::error( 'No groups found.' );
			}

			$option_key_meta = 'nb_report_password';
			$passwords       = [];

			foreach ( $groups['groups'] as $group ) {
				WP_CLI::line( 'Group ID: ' . absint( $group ) );

				$value = groups_get_groupmeta( $group, $option_key_meta, true );

				if ( $value ) {
					$passwords[] = $value;
					WP_CLI::line( 'Password is already set.' );
				} else {
					$password = wp_generate_password( 10, false );
					while ( in_array( $password, $passwords, true ) ) {
						WP_CLI::line( 'Existing password. Create new random one.' );
						$password = wp_generate_password( 10, false );
					}

					groups_update_groupmeta( $group, $option_key_meta, $password );
					$passwords[] = $password;

					WP_CLI::line( 'Password is set.' );
					WP_CLI::line( '----------------' );
				}
			}
		}

		/**
		 * Export passwords
		 * This function is run on the command "wp nb-library export_passwords"
		 */
		public function export_passwords() {
			$query_args = [
				'per_page'    => 5000,
				'group_type'  => 'schoollibrary',
				'show_hidden' => true,
				'order'       => 'ASC',
				'fields'      => 'ids',
			];

			$groups = groups_get_groups( $query_args );
			if ( empty( $groups['groups'] ) ) {
				WP_CLI::error( 'No groups found.' );
			}

			$option_key_password = 'nb_report_password';
			$option_key_meta     = 'nb_library_meta';

			$fh = fopen( 'php://temp', 'rw' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen

			fputs( $fh, $bom = chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

			$counter  = 1;
			$headings = [
				'Teller',
				'Gruppe-ID i Nisje',
				'Biblioteknavn',
				'Passord',
				'E-post',
				'Biblioteknummer',
				'Bibliotektype',
				'Kontaktperson',
				'E-post kontaktperson',
				'Kommunenummer',
				'Fylkesnummer',
			];

			fputcsv( $fh, $headings );

			foreach ( $groups['groups'] as $group ) {
				$line = [
					$counter,
					$group,
					str_replace( '&amp;', '&', wp_strip_all_tags( groups_get_group( $group )->name ) ),
				];

				$password = groups_get_groupmeta( $group, $option_key_password, true );
				$line[]   = $password;

				$value = groups_get_groupmeta( $group, $option_key_meta, true );
				$data  = json_decode( $value );

				$line[] = isset( $data->epost_adr ) ? $data->epost_adr : '';
				$line[] = isset( $data->bibnr ) ? $data->bibnr : '';
				$line[] = isset( $data->bibltype ) ? $data->bibltype : '';
				$line[] = isset( $data->bibleder ) ? $data->bibleder : '';
				$line[] = isset( $data->cepost_adr ) ? $data->cepost_adr : '';
				$line[] = isset( $data->kommnr ) ? $data->kommnr : '';
				if ( isset( $data->kommnr ) ) {
					$line[] = substr( $data->kommnr, 0, 2 );
				} else {
					$line[] = isset( $data->fylknr ) ? substr( $data->fylknr, 0, 2 ) : '';
				}

				$counter++;

				fputcsv( $fh, $line );
			}

			rewind( $fh );
			$csv = stream_get_contents( $fh );
			echo $csv; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		/**
		 * Create passwords
		 * This function is run on the command "wp nb-library create_municipality_users"
		 */
		public function create_municipality_users() {
			$file_user_contents   = file_get_contents( 'users.csv', true );
			$file_mapped_contents = file_get_contents( 'mapped.csv', true );

			$data   = str_getcsv( $file_user_contents, "\n" );
			$mapped = str_getcsv( $file_mapped_contents, "\n" );

			$mapped_array = [];
			foreach ( $mapped as $map_key => $map_row ) {
				$lib_row = str_getcsv( $map_row, ',' );

				$mapped_array[ $lib_row[2] ] = $lib_row[3];
			}

			$counter = 1;
			foreach ( $data as $key => $row ) {
				$user = str_getcsv( $row, ';' );

				$current_user = get_user_by( 'email', $user[2] );
				if ( false === $current_user ) {
					$current_user = get_user_by( 'login', sanitize_user( sanitize_title( $user[1] ), true ) );
				}

				$password = '';
				if ( isset( $mapped_array[ $user[1] ] ) ) {
					$password = $mapped_array[ $user[1] ];
				}

				if ( empty( $password ) ) {
					WP_CLI::line( 'nonmatching password' );
				} elseif ( false !== $current_user ) {
					update_user_meta( $current_user->ID, 'nisje_nb_ar_dashboard_municipality', $user[0] );

					WP_CLI::line( $counter . ',"' . $current_user->ID . '","' . $current_user->user_login . '","existing-user-has-password","' . $current_user->user_email . '","' . $user[2] . '","' . $user[0] . '"' );
				} else {
					$user_data = [
						'user_login' => sanitize_user( sanitize_title( $user[1] ), true ),
						'user_email' => $user[2],
						'user_pass'  => $password,
						'role'       => 'nisje_nb_ar_user_light',
					];

					$user_id = wp_insert_user( $user_data );
					if ( ! is_wp_error( $user_id ) ) {
						update_user_meta( $user_id, 'nisje_nb_ar_dashboard_municipality', $user[0] );
						update_user_meta( $user_id, 'nb_report_user_password', $password );

						WP_CLI::line( $counter . ',"' . $user_id . '","' . sanitize_user( sanitize_title( $user[1] ), true ) . '","' . $password . '","' . $user[2] . '","' . $user[0] . '"' );
					} else {
						WP_CLI::line( 'Import failed for user ' . $user[2] );
					}
				}

				$counter++;
			}

			WP_CLI::line( 'Total ' . $counter );
		}

		/**
		 * Create passwords
		 * This function is run on the command "wp nb-library create_municipality_passwords"
		 */
		public function create_municipality_passwords() {
			$query_args = [
				'per_page'    => 5000,
				'group_type'  => 'municipality',
				'show_hidden' => true,
				'fields'      => 'ids',
			];

			$groups = groups_get_groups( $query_args );
			if ( empty( $groups['groups'] ) ) {
				WP_CLI::error( 'No groups found.' );
			}

			$option_key_meta = 'nb_report_password';
			$passwords       = [];

			foreach ( $groups['groups'] as $group ) {
				WP_CLI::line( 'Group ID: ' . absint( $group ) );

				$value = groups_get_groupmeta( $group, $option_key_meta, true );

				if ( $value ) {
					$passwords[] = $value;
					WP_CLI::line( 'Password is already set.' );
				} else {
					$password = wp_generate_password( 10, false );
					while ( in_array( $password, $passwords, true ) ) {
						WP_CLI::line( 'Existing password. Create new random one.' );
						$password = wp_generate_password( 10, false );
					}

					groups_update_groupmeta( $group, $option_key_meta, $password );
					$passwords[] = $password;

					WP_CLI::line( 'Password is set.' );
					WP_CLI::line( '----------------' );
				}
			}
		}

		/**
		 * Export passwords
		 * This function is run on the command "wp nb-library export_municipality_passwords"
		 */
		public function export_municipality_passwords() {
			$query_args = [
				'per_page'    => 5000,
				'group_type'  => 'municipality',
				'show_hidden' => true,
				'order'       => 'ASC',
				'fields'      => 'ids',
			];

			$groups = groups_get_groups( $query_args );
			if ( empty( $groups['groups'] ) ) {
				WP_CLI::error( 'No groups found.' );
			}

			$option_key_password = 'nb_report_password';
			$option_key_meta     = 'nb_library_meta';

			$fh = fopen( 'php://temp', 'rw' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen

			fputs( $fh, $bom = chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

			$counter  = 1;
			$headings = [
				'Teller',
				'Gruppe-ID i Nisje',
				'Biblioteknavn',
				'Passord',
				'E-post',
				'Biblioteknummer',
				'Bibliotektype',
				'Kontaktperson',
				'E-post kontaktperson',
				'Kommunenummer',
				'Fylkesnummer',
			];

			fputcsv( $fh, $headings );

			foreach ( $groups['groups'] as $group ) {
				$line = [
					$counter,
					$group,
					str_replace( '&amp;', '&', wp_strip_all_tags( groups_get_group( $group )->name ) ),
				];

				$password = groups_get_groupmeta( $group, $option_key_password, true );
				$line[]   = $password;

				$value = groups_get_groupmeta( $group, $option_key_meta, true );
				$data  = json_decode( $value );

				$line[] = isset( $data->epost_adr ) ? $data->epost_adr : '';
				$line[] = isset( $data->bibnr ) ? $data->bibnr : '';
				$line[] = isset( $data->bibltype ) ? $data->bibltype : '';
				$line[] = isset( $data->bibleder ) ? $data->bibleder : '';
				$line[] = isset( $data->cepost_adr ) ? $data->cepost_adr : '';
				$line[] = isset( $data->kommnr ) ? $data->kommnr : '';
				if ( isset( $data->kommnr ) ) {
					$line[] = substr( $data->kommnr, 0, 2 );
				} else {
					$line[] = isset( $data->fylknr ) ? substr( $data->fylknr, 0, 2 ) : '';
				}

				$counter++;

				fputcsv( $fh, $line );
			}

			rewind( $fh );
			$csv = stream_get_contents( $fh );
			echo $csv; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		/**
		 * Create passwords
		 * This function is run on the command "wp nb-library create_countycouncil_users"
		 */
		public function create_countycouncil_users() {
			$file_user_contents = file_get_contents( 'countycouncil.csv', true );
			$data               = str_getcsv( $file_user_contents, "\n" );

			$counter = 1;
			foreach ( $data as $key => $row ) {
				$user = str_getcsv( $row, ',' );

				$current_user = get_user_by( 'email', $user[4] );
				if ( false === $current_user ) {
					$current_user = get_user_by( 'login', sanitize_user( sanitize_title( $user[2] ), true ) );
				}

				$password = $user[3];

				if ( false !== $current_user ) {
					update_user_meta( $current_user->ID, 'nisje_nb_ar_dashboard_countycouncil', $user[10] );

					WP_CLI::line( $counter . ',"' . $current_user->ID . '","' . $current_user->user_login . '","existing-user-has-password","' . $current_user->user_email . '","' . $user[4] . '","' . $user[10] . '"' );
				} else {
					$user_data = [
						'user_login' => sanitize_user( sanitize_title( $user[2] ), true ),
						'user_email' => $user[4],
						'user_pass'  => $password,
						'role'       => 'nisje_nb_ar_user_countycouncil',
					];

					$user_id = wp_insert_user( $user_data );
					if ( ! is_wp_error( $user_id ) ) {
						update_user_meta( $user_id, 'nisje_nb_ar_dashboard_countycouncil', $user[0] );
						update_user_meta( $user_id, 'nb_report_user_password', $password );

						WP_CLI::line( $counter . ',"' . $user_id . '","' . sanitize_user( sanitize_title( $user[2] ), true ) . '","' . $password . '","' . $user[4] . '","' . $user[10] . '"' );
					} else {
						WP_CLI::line( 'Import failed for user ' . $user[2] );
					}
				}

				$counter++;
			}

			WP_CLI::line( 'Total ' . $counter );
		}

		/**
		 * Create passwords
		 * This function is run on the command "wp nb-library create_countycouncil_passwords"
		 */
		public function create_countycouncil_passwords() {
			$query_args = [
				'per_page'    => 5000,
				'group_type'  => 'countycouncil',
				'show_hidden' => true,
				'fields'      => 'ids',
			];

			$groups = groups_get_groups( $query_args );
			if ( empty( $groups['groups'] ) ) {
				WP_CLI::error( 'No groups found.' );
			}

			$option_key_meta = 'nb_report_password';
			$passwords       = [];

			foreach ( $groups['groups'] as $group ) {
				WP_CLI::line( 'Group ID: ' . absint( $group ) );

				$value = groups_get_groupmeta( $group, $option_key_meta, true );

				if ( $value ) {
					$passwords[] = $value;
					WP_CLI::line( 'Password is already set.' );
				} else {
					$password = wp_generate_password( 10, false );
					while ( in_array( $password, $passwords, true ) ) {
						WP_CLI::line( 'Existing password. Create new random one.' );
						$password = wp_generate_password( 10, false );
					}

					groups_update_groupmeta( $group, $option_key_meta, $password );
					$passwords[] = $password;

					WP_CLI::line( 'Password is set.' );
					WP_CLI::line( '----------------' );
				}
			}
		}

		/**
		 * Export passwords
		 * This function is run on the command "wp nb-library export_countycouncil_passwords"
		 */
		public function export_countycouncil_passwords() {
			$query_args = [
				'per_page'    => 5000,
				'group_type'  => 'countycouncil',
				'show_hidden' => true,
				'order'       => 'ASC',
				'fields'      => 'ids',
			];

			$groups = groups_get_groups( $query_args );
			if ( empty( $groups['groups'] ) ) {
				WP_CLI::error( 'No groups found.' );
			}

			$option_key_password = 'nb_report_password';
			$option_key_meta     = 'nb_library_meta';

			$fh = fopen( 'php://temp', 'rw' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen

			fputs( $fh, $bom = chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

			$counter  = 1;
			$headings = [
				'Teller',
				'Gruppe-ID i Nisje',
				'Biblioteknavn',
				'Passord',
				'E-post',
				'Biblioteknummer',
				'Bibliotektype',
				'Kontaktperson',
				'E-post kontaktperson',
				'Kommunenummer',
				'Fylkesnummer',
			];

			fputcsv( $fh, $headings );

			foreach ( $groups['groups'] as $group ) {
				$line = [
					$counter,
					$group,
					str_replace( '&amp;', '&', wp_strip_all_tags( groups_get_group( $group )->name ) ),
				];

				$password = groups_get_groupmeta( $group, $option_key_password, true );
				$line[]   = $password;

				$value = groups_get_groupmeta( $group, $option_key_meta, true );
				$data  = json_decode( $value );

				$line[] = isset( $data->epost_adr ) ? $data->epost_adr : '';
				$line[] = isset( $data->bibnr ) ? $data->bibnr : '';
				$line[] = isset( $data->bibltype ) ? $data->bibltype : '';
				$line[] = isset( $data->bibleder ) ? $data->bibleder : '';
				$line[] = isset( $data->cepost_adr ) ? $data->cepost_adr : '';
				$line[] = isset( $data->kommnr ) ? $data->kommnr : '';
				if ( isset( $data->kommnr ) ) {
					$line[] = substr( $data->kommnr, 0, 2 );
				} else {
					$line[] = isset( $data->fylknr ) ? substr( $data->fylknr, 0, 2 ) : '';
				}

				$counter++;

				fputcsv( $fh, $line );
			}

			rewind( $fh );
			$csv = stream_get_contents( $fh );
			echo $csv; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	$library_meta = new NB_Library();

	WP_CLI::add_command( 'nb-library', $library_meta );
}
