<?php
/**
 * Projects import class.
 *
 * @package Nisje
 */

declare( strict_types=1 );

namespace Dekode\Nisje\Commands;

use WP_CLI;
use Exception;
use StdClass;
use WP_Post;

if ( class_exists( 'WP_CLI' ) ) {
	/**
	 * Projects functionality
	 */
	class NB_Projects {
		/**
		 * Stats
		 *
		 * @var array
		 */
		private $stats = [
			'total'            => 0,
			'insert'           => 0,
			'update'           => 0,
			'0_library'        => 0,
			'1_library'        => 0,
			'multiple_library' => 0,
			'failed'           => 0,
			'groups'           => [
				'0'        => [],
				'1'        => [],
				'multiple' => [],
			],
		];

		/**
		 * Types
		 *
		 * @var array
		 */
		private $types = [
			'Ny søknad',
			'Søknad',
			'Statusrapport',
			'Sluttrapport',
		];

		/**
		 * Categories
		 *
		 * @var array
		 */
		private $categories = [
			'Innsatsområder' => 'innsatsomrader',
			'Frie Forsøk'    => 'frie-forsok',
			'Forprosjekt'    => 'forprosjekt',
		];

		/**
		 * Subtypes
		 *
		 * @var array
		 */
		private $subtypes = [
			'åpen_tilgang_data_vitenskap'         => 'apen-tilgang',
			'biblioteket_som_læringsarena'        => 'biblioteket_som_laeringsarena',
			'biblioteket_som_møteplass'           => 'biblioteket_som_moteplass',
			'digitale_tjenester'                  => 'digitale_tjenester',
			'formidling-bibs'                     => 'formidling-bibs',
			'hverdagsintegrering'                 => 'hverdagsintegrering',
			'mangfold_og_inkludering'             => 'mangfold_og_inkludering',
			'nye_formidlingsmetoder'              => 'nye_formidlingsmetoder',
			'nye_samarbeidsformer_og_partnerskap' => 'nye-sam',
			'nye-sam'                             => 'nye-sam',
			'tjenesteutvikling'                   => 'tjenesteutvikling',
			'utvikling_nye_bibliotektjenester'    => 'utvikling-ny-bibs',
			'utvikling-ny-bibs'                   => 'utvikling-ny-bibs',
			'formidling_av_bibliotekenes_innhold_digitalt_og_fysisk' => 'formidling-bibs',
		];

		/**
		 * Duration
		 *
		 * @var array
		 */
		private $duration = [
			'Ettårig' => 1,
			'Toårig'  => 2,
			'Treårig' => 3,
		];

		/**
		 * This function is run on the command "wp nb-projects convert_projects"
		 */
		public function convert_projects() {
			$query_args = [
				'post_type'      => nisje_get_setting( 'post_type_name', 'nb_applications' ),
				'posts_per_page' => -1,
				'post_status'    => 'any',
				'meta_key'       => 'nb_status', // phpcs:ignore
				'meta_value'     => 'missingstatus', // phpcs:ignore
			];

			$query = new \WP_Query( $query_args );

			if ( $query->have_posts() ) {
				while ( $query->have_posts() ) {
					$query->the_post();

					$post = get_post();
					$args = (array) $post;

					WP_CLI::line( 'Application ID: ' . get_the_ID() );

					$parent_group_id = get_post_meta( get_the_ID(), 'nb_group_id', true );
					$project_group   = get_post_meta( get_the_ID(), '_nisje_nb_project_id', true );

					WP_CLI::line( 'Group ID: ' . $parent_group_id );

					if ( (int) $parent_group_id !== (int) $post->post_parent ) {
						WP_CLI::line( 'Mismatch in group connection ' . $parent_group_id . ' vs ' . $post->post_parent );
					}

					if ( ! $parent_group_id ) {
						WP_CLI::line( 'Group ID: ' . $parent_group_id );
					} elseif ( $project_group ) {
						if ( 0 === $post->post_parent ) {
							$args['post_parent'] = $parent_group_id;
							wp_update_post( $args );
							update_post_meta( $post->ID, '_nisje_group_connection', $parent_group_id );
							WP_CLI::line( 'Connected the post to the group ID ' . $parent_group_id );
						}

						WP_CLI::line( 'Project group already exists ' . $project_group );
					} else {
						// Create group based on convertion.
						$args = [
							'group_id'     => 0,
							'creator_id'   => $post->post_author,
							'name'         => $post->post_title,
							'description'  => ' ',
							'slug'         => '',
							'status'       => 'public',
							'parent_id'    => $parent_group_id,
							'enable_forum' => 0,
						];

						$group_id = groups_create_group( $args );

						if ( ! $group_id ) {
							WP_CLI::line( 'Group not created' );
						} else {
							$group = nisje_get_group( $group_id );
							if ( is_wp_error( $group ) ) {
								return $group;
							}

							bp_groups_set_group_type( $group->id, 'project' );

							groups_update_groupmeta( $group->id, '_nisje_nb_application_id', $post->ID );
							update_post_meta( $post->ID, '_nisje_nb_project_id', $group->id );

							$args['post_parent'] = $parent_group_id;
							wp_update_post( $args );
							update_post_meta( $post->ID, '_nisje_group_connection', $parent_group_id );

							WP_CLI::line( 'Group created with group ID: ' . $group->id );
						}
					}

					if ( 'missingstatus' !== $post->post_status ) {
						$args['post_status'] = 'missingstatus';
						wp_update_post( $args );

						WP_CLI::line( 'Update application with new status' );
					}

					WP_CLI::line( '----------------------------' );
				}
			}
		}

		/**
		 * This function is run on the command "wp nb-projects import_projects"
		 */
		public function import_projects() {
			$dir = dirname( dirname( __FILE__ ) ) . '/import/*.xml';

			$files = glob( $dir );
			WP_CLI::line( count( $files ) . ' files found' );

			$counter         = 0;
			$reference_store = [];

			foreach ( $files as $xml_file ) {
				$json_args = [];
				$post_args = [];

				$xml    = simplexml_load_file( $xml_file );
				$json   = wp_json_encode( $xml );
				$values = json_decode( $json, true );
				if ( ! isset( $values['@attributes']['id'] ) ) {
					WP_CLI::line( 'Customers reference not found for file ' . $xml_file );
				}

				$reference          = $values['@attributes']['id'];
				$customer_reference = $values['meta']['referanse']['kundereferanse'] ?? '';
				$service_type       = $values['meta']['tjeneste']['type'] ?? '';
				$service_title      = $values['meta']['tjeneste']['tittel'] ?? '';
				$service_sub_type   = $values['meta']['referanse']['type'] ?? '';

				$positive = false;
				if ( '' !== $service_sub_type && 'søknad' === $service_sub_type ) {
					$positive = true;
				}

				if ( 'Søknad' === $service_type ) {
					$positive = true;
				}

				if ( false === $positive ) {
					WP_CLI::line( 'Status or End Report ' . $service_type . ' - ' . $service_sub_type );
					continue;
				}

				if ( ! in_array( $reference, $reference_store, true ) ) {
					$reference_store[] = $reference;
				}

				if ( $reference !== $customer_reference ) {
					if ( in_array( $customer_reference, $reference_store, true ) ) {
						WP_CLI::line( 'Status or End Report' );

						continue;
					}
				}

				$innsending = $values['innsending'] ?? null;
				if ( ! $this->is_project( $innsending ) ) {
					continue;
				}

				$application = $innsending['søknad'] ?? null;
				$org         = $innsending['søkerorganisasjon'] ?? null;
				$project     = $innsending['prosjekt'] ?? null;
				$partners    = $innsending['samarbeidspartnere']['samarbeidspartner'] ?? null;
				$plan        = $innsending['tidsplan'] ?? null;
				$budget      = $innsending['budsjett'] ?? null;

				$project_start = $plan['prosjektperiode']['start']['dato'] ?? null;
				$project_end   = $plan['prosjektperiode']['slutt']['dato'] ?? null;

				$json_args = [
					// Categories.
					'type'                      => $this->parse_main_category( $application ),
					'project-type'              => $this->parse_main_category( $application, true ),
					'subtypes'                  => $this->parse_sub_categories( $application ),
					'project-type-alternatives' => $this->parse_sub_categories( $application, true ),
					// Meta.
					'title'                     => $project['tittel'] ?? '',
					'goal'                      => $project['prosjektmål'] ?? '',
					'description'               => $project['sammendrag'] ?? '',
					// Org.
					'org-name'                  => $org['org_navn'] ?? '',
					'org-number'                => $org['org_nummer'] ?? '',
					'contact-fn'                => $org['kontaktperson']['fornavn'] ?? '',
					'contact-ln'                => $org['kontaktperson']['etternavn'] ?? '',
					'contact-phone'             => $org['kontaktperson']['telefon'] ?? '',
					'contact-email'             => $org['kontaktperson']['epostadresse'] ?? '',
					'is-contact-leader'         => $this->validate_contact_leader( $org ),
					// Duration.
					'duration-years'            => $this->parse_duration( $plan ),
					'activities'                => $this->parse_activities( $plan ),
					'plan-other-info'           => is_string( $plan['kommentarer'] ) ? $plan['kommentarer'] : '',
					// Partners.
					'partners'                  => $this->parse_partners( $partners ),
					// Budget.
					'soknadsbelop'              => $values['meta']['arkiv']['journalpost']['søknadsbeløp'] ?? '',
					'budget'                    => $this->parse_budget( $budget ),
					// Period.
					'period'                    => [
						'start' => $this->parse_date( $project_start ),
						'end'   => $this->parse_date( $project_end ),
					],
				];

				if ( 'no' === $json_args['is-contact-leader'] ) {
					$json_args['leader-contact-fn']    = $org['leder']['fornavn'] ?? '';
					$json_args['leader-contact-ln']    = $org['leder']['etternavn'] ?? '';
					$json_args['leader-contact-phone'] = '';
					$json_args['leader-contact-email'] = '';
				}

				$json_args['har-samarbeidspartnere'] = 'yes';
				if ( empty( $json_args['partners'] ) ) {
					$json_args['har-samarbeidspartnere'] = 'no';
				}

				$post_args = [
					'reference'     => $reference,
					'post_author'   => 56,
					'post_title'    => $json_args['title'],
					'post_content'  => $json_args['description'],
					'json_args'     => $json_args,
					'file'          => $xml_file,
					'post_type'     => nisje_get_setting( 'post_type_name', 'nb_applications' ),
					'post_status'   => 'pending',
					'meta'          => $values['meta'],
					'service-title' => $service_title,
				];

				WP_CLI::line( 'Importing project ' . $xml_file . ' (' . $reference . ')' );

				$this->create_or_update( $post_args );

				$counter++;
			}

			WP_CLI::success( $counter . ' projects imported' );
			WP_CLI::line( '------------------' );
			WP_CLI::line( 'Total: ' . $this->stats['total'] );
			WP_CLI::line( 'Insert: ' . $this->stats['insert'] );
			WP_CLI::line( 'Update: ' . $this->stats['update'] );
			WP_CLI::line( 'Failed: ' . $this->stats['failed'] );
			WP_CLI::line( 'Single Library: ' . $this->stats['1_library'] );
			WP_CLI::line( 'Library not found: ' . $this->stats['0_library'] );
			WP_CLI::line( 'Multiple Libraries: ' . $this->stats['multiple_library'] );

			WP_CLI::line( '------------------' );
			WP_CLI::line( '0 libraries found: ' );
			foreach ( $this->stats['groups']['0'] as $stat ) {
				WP_CLI::line( "\t" . $stat );
			}

			WP_CLI::line( '------------------' );
			WP_CLI::line( '1 library found: ' );
			foreach ( $this->stats['groups']['1'] as $stat ) {
				WP_CLI::line( "\t" . $stat );
			}

			WP_CLI::line( '------------------' );
			WP_CLI::line( 'Multiple libraries found: ' );
			foreach ( $this->stats['groups']['multiple'] as $stat ) {
				WP_CLI::line( "\t" . $stat );
			}
		}

		/**
		 * Takes an project and creates a new post for it or updates an existing one.
		 *
		 * @param array $args Data from Imported file.
		 */
		private function create_or_update( array $args ) {
			$original_post = $this->get_existing_post( $args['reference'] );

			if ( is_object( $original_post ) && 'WP_Post' === get_class( $original_post ) ) {
				$args['ID'] = $original_post->ID;
				WP_CLI::line( "\t" . 'Found original post so updating post ' . $original_post->ID );
				$this->stats['update']++;
			} else {
				WP_CLI::line( "\t" . 'No original post found, creating new' );
				$this->stats['insert']++;
			}

			$post_id = wp_insert_post( $args );

			$group_parent_id = $this->map_parent( $args['json_args']['org-name'] );

			$missing_parent = false;
			if ( ! $group_parent_id ) {
				WP_CLI::line( "\t" . 'Could not find library' );
				$group_parent_id = 0;
			}

			$args['post_parent'] = $group_parent_id;

			$post_id = wp_insert_post( $args );

			if ( $missing_parent ) {
				update_post_meta( $post_id, '_missing_group_id', 1 );
			} else {
				update_post_meta( $post_id, 'nb_group_id', $group_parent_id );
				update_post_meta( $post_id, '_nisje_group_connection', $group_parent_id );
				update_post_meta( $post_id, 'nb_parent_group_id', $group_parent_id );
			}

			update_post_meta( $post_id, '_id_in_remote', $args['reference'] );

			if ( isset( $args['json_args']['period'] ) ) {
				update_post_meta( $post_id, 'nb_project_period', $args['json_args']['period'] );
			}

			if ( isset( $args['json_args']['period']['start'] ) ) {
				update_post_meta( $post_id, 'nb_project_start', $args['json_args']['period']['start'] );
			}

			if ( isset( $args['json_args']['period']['end'] ) ) {
				update_post_meta( $post_id, 'nb_project_end', $args['json_args']['period']['end'] );
			}

			$this->update_taxonomies( $args, $post_id );

			update_post_meta( $post_id, 'nb_json', wp_slash( wp_json_encode( $args['json_args'] ) ) );
			update_post_meta( $post_id, 'nb_project_meta', wp_slash( wp_json_encode( $args['meta'] ) ) );

			if ( $group_parent_id ) {
				// phpcs:disable Squiz.PHP.CommentedOutCode.Found
				$parent_group_admins = groups_get_group_admins( $group_parent_id );

				$creator_id = 56;
				if ( isset( $parent_group_admins[0]->user_id ) ) {
					$creator_id = $parent_group_admins[0]->user_id;
				}

				$group_id = get_post_meta( $post_id, 'nb_project_group_id', true );

				$group_args = [
					'group_id'     => $group_id,
					'creator_id'   => $creator_id,
					'name'         => $args['post_title'],
					'description'  => ' ',
					'status'       => 'private',
					'parent_id'    => $group_parent_id,
					'enable_forum' => 0,
				];

				$group_id = groups_create_group( $group_args );

				if ( ! $group_id ) {
					return;
				}

				// Add user to group.
				foreach ( $parent_group_admins as $parent_group_admin ) {
					groups_join_group( $group_id, $parent_group_admin->user_id );
					groups_promote_member( $parent_group_admin->user_id, $group_id, 'admin' );
				}

				bp_groups_set_group_type( $group_id, 'project' );
				groups_update_groupmeta( $group_id, 'nb_application_id', $post_id );
				groups_update_groupmeta( $group_id, 'nb_json', wp_slash( wp_json_encode( $args['json_args'] ) ) );

				update_post_meta( $post_id, 'nb_project_group_id', $group_id );
			}

			$this->stats['total']++;
		}

		/**
		 * Finds existing post in WordPress
		 *
		 * @param string $remote_id The ID of the post to be found.
		 *
		 * @return array|boolean
		 */
		private function get_existing_post( string $remote_id ) {
			$args = [
				'post_type'   => nisje_get_setting( 'post_type_name', 'nb_applications' ),
				'post_status' => nisje_nb_get_post_statuses(),
				'meta_query'  => [ // phpcs:ignore
					[
						'key'     => '_id_in_remote',
						'value'   => $remote_id,
						'compare' => '=',
					],
				],
			];

			$query = new \WP_Query( $args );

			$posts = $query->get_posts();
			if ( isset( $posts[0] ) ) {
				WP_CLI::line( 'Post found: ' . $posts[0]->post_title );
				WP_CLI::line( 'Post found: ' . $posts[0]->ID );
				return $posts[0];
			} else {
				return null;
			}
		}

		/**
		 * Update the taxonomies of this project with the values sent
		 *
		 * @param array $args    The project from the WP REST API with the taxonomies.
		 * @param int   $post_id The ID of the post that should have these taxonomies.
		 */
		private function update_taxonomies( array $args, int $post_id ) {
			wp_set_object_terms( $post_id, [ 'project' ], nisje_get_setting( 'taxonomy_type_name', 'nb_applications' ) );

			$category = [ $args['json_args']['type'] ];
			wp_set_object_terms( $post_id, $category, nisje_get_setting( 'taxonomy_category_name', 'nb_applications' ) );

			$subcategories = $args['json_args']['subtypes'];
			wp_set_object_terms( $post_id, $subcategories, nisje_get_setting( 'taxonomy_subcategory_name', 'nb_applications' ) );

			$period     = $args['json_args']['period'];
			$start_year = (int) gmdate( 'Y', strtotime( $period['start'] ) );
			$end_year   = (int) gmdate( 'Y', strtotime( $period['end'] ) );

			$years = range( $start_year, $end_year );
			wp_set_object_terms( $post_id, array_map( 'strval', $years ), nisje_get_setting( 'taxonomy_year_name', 'nb_applications' ) );

			if ( ! empty( $args['service-title'] ) ) {
				wp_set_object_terms( $post_id, $args['service-title'], nisje_get_setting( 'taxonomy_application_name', 'nb_applications' ) );
			}
		}

		/**
		 * Find library
		 *
		 * @param string $library Library name.
		 * @return int Id of Group.
		 */
		private function map_parent( string $library ) : int {
			$groups = groups_get_groups( [
				'search_terms' => $library,
				'group_type'   => 'library',
			] );

			if ( 0 === $groups['total'] ) {
				WP_CLI::line( "\t" . 'Found 0 libraries' );
				$this->stats['0_library']++;
				$this->stats['groups']['0'][] = $library;
			} elseif ( 1 === $groups['total'] ) {
				WP_CLI::line( "\t" . 'Found 1 library' );
				WP_CLI::line( "\t" . 'Library: ' . $library );
				WP_CLI::line( "\t" . 'Group: ' . $groups['groups'][0]->name . ', ' . $groups['groups'][0]->id );
				WP_CLI::line( "\t" . '--------------------------' );

				$this->stats['1_library']++;
				$this->stats['groups']['1'][] = $library . ' ---- ' . $groups['groups'][0]->name;

				return $groups['groups'][0]->id;
			} else {
				WP_CLI::line( "\t" . 'Found ' . $groups['total'] . ' libraries' );
				WP_CLI::line( "\t" . 'Library: ' . $library );
				foreach ( $groups['groups'] as $group ) {
					WP_CLI::line( "\t" . 'Group: ' . $group->name . ', ' . $group->id );
				}
				$this->stats['multiple_library']++;
				$this->stats['groups']['multiple'][] = $library;
			}

			return 0;
		}

		/**
		 * Adds the hostname and scheme to relative URLs.
		 *
		 * @param string $text The string to perform the replacement in.
		 *
		 * @return string
		 */
		private function make_urls_absolute( string $text ): string {
			$text = preg_replace( '/<img(.*)src="\/(.*)"(.*)>/u', '<img\\1src="' . $this->api_url . '/\\2"\\3"', $text );

			return $text;
		}

		/**
		 * Parse main category
		 *
		 * @param array $application Application.
		 * @param bool  $map         Map values.
		 *
		 * @return string Category.
		 */
		private function parse_main_category( $application, $map = false ) {
			if ( isset( $application['kategori']['@attributes']['type'] ) ) {
				if ( $map ) {
					if ( isset( $this->categories[ $application['kategori']['@attributes']['type'] ] ) ) {
						return $this->categories[ $application['kategori']['@attributes']['type'] ];
					}
				} else {
					return $application['kategori']['@attributes']['type'];
				}
			}

			return '';
		}

		/**
		 * Parse main category
		 *
		 * @param array $application Application.
		 * @param bool  $map         Map values.
		 *
		 * @return array Subcategories.
		 */
		private function parse_sub_categories( $application, $map = false ) {
			$retval = [];
			if ( isset( $application['kategori']['underkategori'] ) ) {
				foreach ( $application['kategori']['underkategori'] as $key => $value ) {
					if ( 'ja' === $value ) {
						if ( $map ) {
							if ( isset( $this->subtypes[ $key ] ) ) {
								$retval[] = $this->subtypes[ $key ];
							}
						} else {
							$retval[] = $key;
						}
					}
				}
			}

			return $retval;
		}

		/**
		 * Validate contact leader.
		 *
		 * @param array $org Org data.
		 * @return string Parsed data.
		 */
		private function validate_contact_leader( $org ) {
			$fn = $org['kontaktperson']['fornavn'] ?? '';
			$ln = $org['kontaktperson']['etternavn'] ?? '';

			$lfn = $org['leder']['fornavn'] ?? '';
			$lln = $org['leder']['etternav'] ?? '';

			$retval = 'no';
			if ( $fn === $ln && $ln === $lfn ) {
				$retval = 'yes';
			}

			return $retval;
		}

		/**
		 * Parse duration.
		 *
		 * @param array $plan Plan data.
		 * @return string Duration
		 */
		private function parse_duration( $plan ) {
			$retval = 3;

			if ( isset( $this->duration[ $plan['@attributes']['prosjektvarighet'] ] ) ) {
				$retval = $this->duration[ $plan['@attributes']['prosjektvarighet'] ];
			}

			return $retval;
		}

		/**
		 * Parse activities
		 *
		 * @param array $plan Plan data.
		 * @return string Parsed data.
		 */
		private function parse_activities( $plan ) {
			$retval = [];

			if ( isset( $plan['aktiviteter']['aktivitet'] ) && is_array( $plan['aktiviteter']['aktivitet'] ) ) {
				foreach ( $plan['aktiviteter']['aktivitet'] as $activity ) {
					$start_date = $activity['start']['dato'] ?? '';
					$end_date   = $activity['slutt']['dato'] ?? '';

					$retval[] = [
						'description' => $activity['beskrivelse'] ?? '',
						'datepicker'  => [
							'startDate' => $this->parse_date( $start_date ),
							'endDate'   => $this->parse_date( $end_date ),
						],
					];
				}
			}

			return $retval;
		}

		/**
		 * Parse partner
		 *
		 * @param array $partner Partner.
		 * @return array Parse partner
		 */
		private function parse_partner( $partner ) {
			return [
				'name'        => $partner['navn'] ?? '',
				'description' => $partner['rolle']['sammendrag'] ?? '',
				'contact'     => [
					'address'    => $partner['kontaktinformasjon']['adresse'] ?? '',
					'zip'        => $partner['kontaktinformasjon']['postnummer'] ?? '',
					'location'   => $partner['kontaktinformasjon']['poststed'] ?? '',
					'first_name' => $partner['kontaktperson']['fornavn'] ?? '',
					'last_name'  => $partner['kontaktperson']['etternavn'] ?? '',
					'phone'      => $partner['kontaktperson']['telefon'] ?? '',
					'email'      => $partner['kontaktperson']['epostadresse'] ?? '',
				],
			];
		}

		/**
		 * Parse partners
		 *
		 * @param array $partners Partners data.
		 * @return string Parsed data.
		 */
		private function parse_partners( $partners ) {
			$retval = [];

			if ( isset( $partners['navn'] ) ) {
				$retval[] = $this->parse_partner( $partners );
			} else {
				if ( is_array( $partners ) ) {
					foreach ( $partners as $partner ) {
						$retval[] = $this->parse_partner( $partner );
					}
				}
			}

			return $retval;
		}

		/**
		 * Parse budget
		 *
		 * @param array $budget Budget data.
		 * @return string Parsed data.
		 */
		private function parse_budget( $budget ) {
			$expense_period = [];
			$income_period  = [];

			if ( is_array( $budget['utgifter']['utgift'] ) ) {
				foreach ( $budget['utgifter']['utgift'] as $expense ) {
					if ( isset( $expense['periode'][0] ) ) {
						$description = $expense['beskrivelse'];
						foreach ( $expense['periode'] as $exp ) {
							if ( 0 !== (int) $exp['beløp'] ) {
								$expense_period[ 'expenses-' . $exp['@attributes']['nr'] ][] = [
									'sum'         => $exp['beløp'],
									'description' => $description,
								];
							}
						}
					} elseif ( isset( $expense['periode']['@attributes']['nr'] ) ) {
						if ( 0 !== (int) $expense['periode']['beløp'] ) {
							$expense_period[ 'expenses-' . $expense['periode']['@attributes']['nr'] ][] = [
								'sum'         => $expense['periode']['beløp'],
								'description' => $expense['beskrivelse'],
							];
						}
					}
				}
			}

			if ( is_array( $budget['inntekter']['inntekt'] ) ) {
				foreach ( $budget['inntekter']['inntekt'] as $income ) {
					if ( isset( $income['periode'][0] ) ) {
						$description = $income['beskrivelse'];
						foreach ( $income['periode'] as $inc ) {
							if ( 0 !== (int) $inc['beløp'] ) {
								$income_period[ 'income-' . $inc['@attributes']['nr'] ][] = [
									'sum'         => $inc['beløp'],
									'description' => $description,
								];
							}
						}
					} elseif ( isset( $income['periode']['@attributes']['nr'] ) ) {
						if ( 0 !== (int) $income['periode']['beløp'] ) {
							$income_period[ 'income-' . $income['periode']['@attributes']['nr'] ][] = [
								'sum'         => $income['periode']['beløp'],
								'description' => $income['beskrivelse'],
							];
						}
					}
				}
			}

			return array_merge( $income_period, $expense_period );
		}

		/**
		 * Parse dates
		 *
		 * @param string $date Date.
		 * @return string Parsed date.
		 */
		private function parse_date( $date ) {
			$time = strtotime( $date );
			if ( $time ) {
				return gmdate( DATE_ATOM, $time );
			}

			return $date;
		}

		/**
		 * Is project?
		 *
		 * @param array $type Type.
		 * @return boolean If is project or not.
		 */
		private function is_project( $type ) {
			if ( isset( $type['@attributes']['type'] ) ) {
				if ( in_array( $type['@attributes']['type'], [ 'Søknad' ], true ) ) {
					return true;
				}
			}

			return false;
		}

		/**
		 * Format string.
		 *
		 * @param string $str String.
		 * @return string Formatted string.
		 */
		private function format_string( string $str ) : string {
			$str = str_replace( '”', '"', $str );

			return $str;
		}
	}

	$projects = new NB_Projects();

	WP_CLI::add_command( 'nb-projects', $projects );
}
