<?php
/**
 * Profile layout helper functions.
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Register Profile Layouts
 *
 * @param string $key          Key.
 * @param string $name         Name.
 * @param string $label        Label.
 * @param string $instructions Description.
 *
 * @return object Profile Layout object
 */
function nisje_register_profile_layout( $key, $name, $label, $instructions = '' ) {
	global $nisje_profile_layouts;

	if ( ! is_array( $nisje_profile_layouts ) ) {
		$nisje_profile_layouts = [];
	}

	if ( empty( $key ) || empty( $key ) || empty( $label ) ) {
		return false;
	}

	$nisje_profile_layout_object = new Nisje_Profile_Layout( $key, $name, $label, $instructions );

	$nisje_profile_layouts[] = $nisje_profile_layout_object;

	do_action( 'nisje_registered_profile_layout_object', $nisje_profile_layout_object );

	return $nisje_profile_layout_object;
}

/**
 * Get Registered Profile Layouts.
 *
 * @return array Array of registered profile layouts.
 */
function nisje_get_profile_layouts() {
	global $nisje_profile_layouts;

	if ( ! is_array( $nisje_profile_layouts ) ) {
		$nisje_profile_layouts = [];
	}

	return $nisje_profile_layouts;
}

/**
 * Get layouts for a specific user.
 *
 * @param int    $user_id       User ID.
 * @param string $layout_filter Get specific layout.
 *
 * @return array Array of registered profile layouts.
 */
function nisje_get_user_profile_layouts( $user_id, $layout_filter = 'all' ) {
	$cache_key = nisje_get_setting( 'cache_key', 'profile_layouts' );

	$cache_user_key = 'user_' . $user_id . '_' . $cache_key;

	$retval = wp_cache_get( $cache_user_key, 'di_profile_fields' );
	if ( false === $retval ) {
		$profile_layouts = nisje_get_setting( 'layouts', 'profile_layouts' );

		if ( ! empty( $profile_layouts ) ) {
			foreach ( $profile_layouts as $layout => $fields ) {
				foreach ( $fields as $field ) {
					$ret_field = '';

					if ( 'di_profile_layout' === $field['type'] ) {
						$profile_field = new \BP_XProfile_Field( $field['field'], $user_id );
						if ( isset( $profile_field->name ) ) {
							$ret_field = [
								'name'  => $profile_field->name,
								'value' => isset( $profile_field->data->value ) ? xprofile_get_field_data( $field['field'], $user_id ) : '',
								'type'  => $profile_field->type,
							];
						}
					} elseif ( 'di_wp_layout' === $field['type'] ) {
						$user_info = get_userdata( $user_id );
						if ( 'user_email' === $field['field'] ) {
							$ret_field = [
								'name'  => esc_html__( 'E-mail', 'nisje' ),
								'value' => isset( $user_info->user_email ) ? $user_info->user_email : '',
								'type'  => 'email',
							];
						} elseif ( 'description' === $field['field'] ) {
							$ret_field = [
								'name'  => esc_html__( 'Description', 'nisje' ),
								'value' => isset( $user_info->description ) ? $user_info->description : '',
								'type'  => 'text',
							];
						} elseif ( 'nickname' === $field['field'] ) {
							$ret_field = [
								'name'  => esc_html__( 'Nickname', 'nisje' ),
								'value' => isset( $user_info->nickname ) ? $user_info->user_nickname : '',
								'type'  => 'string',
							];
						} elseif ( 'user_nicename' === $field['field'] ) {
							$ret_field = [
								'name'  => esc_html__( 'Nicename', 'nisje' ),
								'value' => isset( $user_info->user_nicename ) ? $user_info->user_nicename : '',
								'type'  => 'string',
							];
						}
					} elseif ( 'di_custom_layout' === $field['type'] ) {
						$ret_field = apply_filters( "nisje_custom_profile_layout_value_{$field['field']}", $field, $user_id );
						if ( ! is_array( $ret_field ) || ! isset( $ret_field['name'], $ret_field['value'] ) ) {
							$ret_field = '';
						}
					}

					$ret_field = apply_filters( 'nisje_profile_layouts_field_before_prepend', $ret_field, $field, $layout );

					$prepend = isset( $field['prepend'] ) ? esc_html( $field['prepend'] ) : '';
					if ( ! empty( $prepend ) && isset( $ret_field['value'] ) && ! is_array( $ret_field['value'] ) ) {
						if ( ( isset( $ret_field['type'] ) && 'title' === $ret_field['type'] ) || ! empty( $ret_field['value'] ) ) {
							$ret_field['value'] = $prepend . ' ' . $ret_field['value'];
						}
					}

					$ret_field = apply_filters( 'nisje_profile_layouts_field', $ret_field, $field, $layout );
					$ret_field = apply_filters_deprecated( 'di_profile_layouts_field', [ $ret_field, $field, $layout ], '1.0.27', 'nisje_profile_layouts_field' );

					if ( ! empty( $ret_field ) ) {
						$ret_field['format'] = $field['format'];

						$retval[ $layout ][] = $ret_field;
					}
				}
			}
		}

		wp_cache_set( $cache_user_key, $retval, 'di_profile_fields', 3600 );
	}

	if ( 'all' !== $layout_filter ) {
		if ( isset( $retval[ $layout_filter ] ) ) {
			return $retval[ $layout_filter ];
		} else {
			return [];
		}
	}

	return $retval;
}
