<?php
/**
 * Nisje Settings API.
 *
 * @package Nisje
 */

namespace Dekode\Nisje;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Nisje Settings API class.
 */
class Settings {
	/**
	 * Settings.
	 *
	 * @var array
	 */
	private $settings = [];

	/**
	 * Constructor.
	 */
	public function __construct() {}

	/**
	 * Main Settings Instance.
	 *
	 * @static object $instance
	 * @return Settings|null The object.
	 */
	public static function instance() {
		// Store the instance locally to avoid private static replication.
		static $instance = null;

		// Only run these methods if they haven't been run previously.
		if ( null === $instance ) {
			$instance = new Settings();
			$instance->setup_globals();
			$instance->setup_actions();
		}

		// Always return the instance.
		return $instance;
	}

	/**
	 * Global variables.
	 */
	private function setup_globals() {
		// Add core setting.
		$this->settings = [
			'core' => [
				'name'         => esc_html__( 'Nisje Community', 'nisje' ),
				'settings_cap' => 'manage_options',
				'settings_key' => 'nisje',
				'roles'        => [
					'administrator',
					'editor',
					'author',
					'contributor',
					'subscriber',
				],
			],
		];
	}

	/**
	 * Constructor. Sets up filters and actions.
	 */
	private function setup_actions() {
		add_action( 'nisje_components_registered', [ $this, 'extend_settings' ] );
		add_action( 'init', [ $this, 'add_public_component_settings' ], 9999 );

		if ( is_admin() ) {
			add_action( 'init', [ $this, 'add_menu' ] );
			add_action( 'acf/include_fields', [ $this, 'register_field_groups' ] );
			add_action( 'acf/save_post', [ $this, 'populate_acf_settings' ], 99 );
		}
	}

	/**
	 * Populate settings from subcontent.
	 */
	public function extend_settings() {
		$settings = apply_filters( 'nisje_extend_settings', [] );

		$this->settings = array_merge( $settings, (array) $this->settings );
	}

	/**
	 * Populate public settings from subcontent.
	 */
	public function add_public_component_settings() {
		do_action( 'nisje_add_public_component_settings' );
	}

	/**
	 * This function will return a value from the settings array.
	 *
	 * @param string           $name  The settings name.
	 * @param string           $group The group name.
	 * @param string|int|Array $value Default fallback value.
	 * @return $value
	 */
	public function get_setting( $name, $group = '', $value = null ) {
		if ( ! empty( $group ) && isset( $this->settings[ $group ][ $name ] ) ) {
			$value = $this->settings[ $group ][ $name ];
		} else {
			if ( isset( $this->settings[ $name ] ) ) {
				$value = $this->settings[ $name ];
			}
		}

		return $value;
	}

	/**
	 * Setup the admin page.
	 */
	public function add_menu() {
		if ( function_exists( 'acf_add_options_page' ) && function_exists( 'acf_add_options_sub_page' ) ) {
			acf_add_options_page( [
				'page_title' => esc_html__( 'Nisje', 'nisje' ),
				'menu_title' => esc_html__( 'Nisje', 'nisje' ),
				'capability' => nisje_get_setting( 'settings_cap', 'core' ),
				'menu_slug'  => nisje_get_setting( 'settings_key', 'core' ),
				'position'   => false,
				'icon_url'   => 'dashicons-admin-settings',
				'redirect'   => false,
			] );

			do_action( 'nisje_settings_add_option_page' );
		}
	}

	/**
	 * Add core fields groups.
	 */
	public function register_field_groups() {
		do_action( 'nisje_settings_register_field_groups' );
	}

	/**
	 * Populate ACF Options Group
	 *
	 * @param string $type Settings type.
	 */
	public function populate_acf_settings( $type ) {
		if ( 'options' === $type ) {
			$screen = get_current_screen();

			do_action( 'nisje_settings_populate_options_from_acf', $screen );
		}
	}
}
