<?php
/**
 * Global functions
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Get plugin version number
 *
 * @param string $type Version type.
 */
function nisje_get_version( $type = 'plugin_version' ) {
	if ( in_array( $type, [ 'plugin_version', 'db_version' ], true ) ) {
		return nisje()->versions[ $type ];
	} else {
		return false;
	}
}

/**
 * Get Setting
 *
 * @param string $name     Setting name.
 * @param string $group    Setting group.
 * @param mixed  $fallback Setting fallback.
 * @return mixed Setting
 */
function nisje_get_setting( $name, $group = false, $fallback = null ) {
	return nisje()->settings->get_setting( $name, $group, $fallback );
}

/**
 * Return path inside plugin folder
 *
 * @param string $path Relative path from plugin root.
 * @return string Full path
 */
function nisje_get_path( $path ) {
	return nisje()->path . $path;
}

/**
 * Return dir inside plugin folder
 *
 * @param string $path Relative dir from plugin root.
 * @return string Full dir
 */
function nisje_get_dir( $path ) {
	return nisje()->dir . $path;
}

/**
 * Check if exists and include
 *
 * @param string $file File path.
 */
function nisje_include( $file ) {
	$path = nisje_get_path( $file );

	if ( file_exists( $path ) ) {
		include_once $path;
	}
}

/**
 * Register block.
 *
 * @param string $file_or_folder Path to the JSON file with metadata definition
 *                               for the block or path to the folder where the
 *                               `block.json` file is located.
 * @param array  $args           Optional. Array of block type arguments.
 *                               Accepts any public property of `WP_Block_Type`.
 */
function nisje_register_block( string $file_or_folder, array $args = [] ): void {
	$filename      = 'block.json';
	$metadata_file = ( substr( $file_or_folder, -strlen( $filename ) ) !== $filename ) ?
		trailingslashit( $file_or_folder ) . $filename :
		$file_or_folder;

	if ( ! file_exists( $metadata_file ) ) {
		return;
	}

	$metadata = json_decode( file_get_contents( $metadata_file ), true ); // phpcs:ignore
	if ( ! is_array( $metadata ) || empty( $metadata['name'] ) ) {
		return;
	}

	// Registers the block on server.
	add_action( 'init', function() use ( $file_or_folder, $args ) {
		register_block_type_from_metadata( $file_or_folder, $args );
	} );

	add_filter( 'allowed_block_types', function( $allowed_blocks ) use ( $metadata ) {
		$allowed_blocks[] = $metadata['name'];
		return $allowed_blocks;
	}, 20 );
}

/**
 * Register Component
 *
 * @param object $component Component object.
 * @return void
 */
function nisje_register_component( $component ) {
	if ( did_action( 'nisje_components_registered' ) ) {
		_doing_it_wrong( __METHOD__, esc_html__( 'Nisje components have already been registered. Please run nisje_register_component() on action nisje_register_components.', 'nisje' ), '1.0.0' );
	}

	add_filter( 'nisje_components', function( $components ) use ( $component ) {
		$components[] = $component;

		return $components;
	} );
}

/**
 * Return component.
 *
 * @param string $key Component key.
 * @return array Active components
 */
function nisje_get_component( $key ) {
	if ( ! did_action( 'nisje_components_registered' ) ) {
		_doing_it_wrong( __METHOD__, esc_html__( 'Nisje components not yet registered.', 'nisje' ), '1.0.0' );
	}

	$key = sanitize_key( $key );
	if ( isset( nisje()->components[ $key ] ) ) {
		return nisje()->components[ $key ];
	}

	return false;
}

/**
 * Fetch data from the rest api
 *
 * @param string $url          Rest url.
 * @param string $query_params Query params.
 */
function nisje_get_api_data( $url, $query_params = [] ) {
	$request = new \WP_REST_Request( 'GET', $url );

	if ( ! empty( $query_params ) ) {
		$request->set_query_params( $query_params );
	}

	$response = rest_do_request( $request );
	$server   = rest_get_server();
	$content  = $server->response_to_data( $response, false );

	return $content;
}
