<?php
/**
 * Global update functions.
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Plugin activation.
 */
function nisje_activation() {
	global $wp_roles, $wpdb;

	if ( class_exists( 'WP_Roles' ) && ! isset( $wp_roles ) ) {
		$wp_roles = new WP_Roles(); // phpcs:ignore
	}

	if ( is_object( $wp_roles ) ) {
		// Create API Role and map administrator to this role.
		$adm_role = $wp_roles->get_role( 'administrator' );
		remove_role( 'nisje_api_user' );
		add_role( 'nisje_api_user', esc_html__( 'Nisje API User', 'nisje' ), $adm_role->capabilities );

		$all_roles   = nisje_get_setting( 'roles', 'core' );
		$all_roles[] = 'nisje_api_user';

		$admin_roles = [
			'administrator',
			'editor',
			'nisje_api_user',
		];

		foreach ( (array) $all_roles as $core_role ) {
			$role = get_role( $core_role );
			if ( $role ) {
				// General.
				$role->remove_cap( 'nisje_administrator' );
				$role->add_cap( 'nisje_read' );

				// Groups.
				$role->add_cap( 'nisje_list_groups' );
				$role->add_cap( 'nisje_list_group_description' );
				$role->add_cap( 'nisje_list_group_activity' );
				$role->add_cap( 'nisje_list_group_members' );
				$role->add_cap( 'nisje_create_groups' );
				$role->add_cap( 'nisje_create_groups_public' );
				$role->add_cap( 'nisje_create_groups_private' );
				$role->add_cap( 'nisje_delete_groups' );
				$role->add_cap( 'nisje_update_groups' );

				// Members.
				$role->add_cap( 'nisje_list_members' );

				// Activity.
				$role->add_cap( 'nisje_activity_create' );
				$role->add_cap( 'nisje_activity_delete' );

				if ( 'nisje_api_user' === $role->name ) {
					$role->add_cap( 'nisje_super_administrator' );
					$role->add_cap( 'nisje_administrator' );
					$role->add_cap( 'nisje_create_groups_hidden' );
					$role->add_cap( 'nisje_create_members' );
					$role->add_cap( 'nisje_delete_members' );
					$role->add_cap( 'nisje_update_members' );

					$role->add_cap( 'nisje_link_users' );
				}

				if ( 'administrator' === $role->name ) {
					$role->add_cap( 'nisje_administrator' );
					$role->add_cap( 'nisje_create_groups_hidden' );
					$role->add_cap( 'nisje_create_members' );
					$role->add_cap( 'nisje_delete_members' );
					$role->add_cap( 'nisje_update_members' );

					$role->add_cap( 'nisje_link_users' );
				}

				if ( 'editor' === $role->name ) {
					// Allow editors to edit options.
					$role->add_cap( 'edit_theme_options' );

					// Allow editors to handle users.
					$role->add_cap( 'create_users' );
					$role->add_cap( 'delete_users' );
					$role->add_cap( 'edit_users' );
					$role->add_cap( 'remove_users' );
					$role->add_cap( 'promote_users' );
					$role->add_cap( 'list_users' );

					$role->add_cap( 'nisje_create_groups_hidden' );
					$role->add_cap( 'nisje_create_members' );
					$role->add_cap( 'nisje_delete_members' );
					$role->add_cap( 'nisje_update_members' );

					$role->add_cap( 'nisje_link_users' );
				}
			}
		}

		// Need to move this logic to the actual component.
		nisje_activate_roles( $admin_roles, nisje_get_setting( 'capability_key', 'global_message' ) );
		nisje_activate_roles( $admin_roles, nisje_get_setting( 'capability_key', 'news' ) );
		nisje_activate_roles( $admin_roles, nisje_get_setting( 'capability_key_group', 'news' ) );
		nisje_activate_roles( $admin_roles, nisje_get_setting( 'capability_key', 'wiki' ) );
		nisje_activate_roles( $admin_roles, nisje_get_setting( 'capability_key_group', 'wiki' ) );
		nisje_activate_roles( $all_roles, nisje_get_setting( 'capability_key', 'event' ) );

		nisje_deactivate_roles( $all_roles, nisje_get_setting( 'capability_key_group', 'event' ) );
		nisje_activate_roles( $admin_roles, nisje_get_setting( 'capability_key_group', 'event' ) );
	}

	// DB specific.
	if ( get_option( 'nisje_db_version' ) !== nisje_get_version( 'db_version' ) ) {
		require_once ABSPATH . '/wp-admin/includes/upgrade.php';

		$charset_collate = '';
		if ( ! empty( $wpdb->charset ) ) {
			$charset_collate = "DEFAULT CHARACTER SET $wpdb->charset";
		}

		if ( ! empty( $wpdb->collate ) ) {
			$charset_collate .= " COLLATE $wpdb->collate";
		}

		$sql = 'CREATE TABLE IF NOT EXISTS ' . $wpdb->di_event_attending . " (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			event_id bigint(20) unsigned NOT NULL,
			user_id bigint(20) unsigned NOT NULL,
			inviter_id bigint(20) unsigned NOT NULL,
			status varchar(20) NOT NULL DEFAULT 'attending',
			date_created datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
			date_modified datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
			PRIMARY KEY (id),
			KEY attending (event_id, user_id, status)
		) $charset_collate;";
		dbDelta( $sql );

		update_option( 'nisje_db_version', nisje_get_version( 'db_version' ) );
	}

	// Options.
	update_option( 'nisje_version', nisje_get_version( 'plugin_version' ) );

	// Reset default options.
	update_option( 'large_size_w', 880 );
	update_option( 'large_size_h', 1100 );
	update_option( 'medium_size_w', 400 );
	update_option( 'medium_size_h', 500 );

	if ( ! get_option( 'nisje_settings_general' ) ) {
		$old_general = get_option( 'dekode_intranet_settings_general' );
		add_option( 'nisje_settings_general', $old_general );
		// Keep old settings for now. Should be removed.
	}

	do_action( 'nisje_component_updates', nisje_get_version( 'plugin_version' ) );

	flush_rewrite_rules();
}

/**
 * Plugin activation.
 */
function nisje_deactivation() {
	// Clean up roles and settings.
}

/**
 * Add post roles.
 *
 * @param array  $roles Roles.
 * @param string $type  Type.
 */
function nisje_activate_roles( $roles, $type ) {
	global $wp_roles, $wpdb;

	if ( ! is_array( $roles ) || empty( $type ) ) {
		return;
	}

	if ( class_exists( 'WP_Roles' ) && ! isset( $wp_roles ) ) {
		$wp_roles = new WP_Roles(); // phpcs:ignore
	}

	if ( is_object( $wp_roles ) ) {
		if ( ! is_array( $roles ) || empty( $type ) ) {
			return;
		}

		$cap_type  = $type;
		$cap_types = $cap_type . 's';

		foreach ( $roles as $core_role ) {
			$role = get_role( $core_role );

			if ( $role ) {

				$role->add_cap( 'read_' . $cap_type );
				$role->add_cap( 'edit_' . $cap_type );
				$role->add_cap( 'delete_' . $cap_type );

				$role->add_cap( 'edit_' . $cap_types );
				$role->add_cap( 'publish_' . $cap_types );
				$role->add_cap( 'delete_published_' . $cap_types );
				$role->add_cap( 'edit_published_' . $cap_types );
				$role->add_cap( 'create_' . $cap_types );
				$role->add_cap( 'upload_files_' . $cap_type );

				if ( in_array( $core_role, [ 'administrator', 'editor' ], true ) ) {
					$role->add_cap( 'edit_others_' . $cap_types );
					$role->add_cap( 'read_private_' . $cap_types );
					$role->add_cap( 'delete_' . $cap_types );
					$role->add_cap( 'delete_private_' . $cap_types );
					$role->add_cap( 'delete_others_' . $cap_types );
					$role->add_cap( 'edit_private_' . $cap_types );
					$role->add_cap( 'use_others_files_' . $cap_type );
					$role->add_cap( 'delete_others_files_' . $cap_type );
				}
			}
		}
	}
}

/**
 * Remove post roles.
 *
 * @param array  $roles Roles.
 * @param string $type  Type.
 */
function nisje_deactivate_roles( $roles, $type ) {
	global $wp_roles, $wpdb;

	// Make sure parameters are correct type and present.
	if ( ! is_array( $roles ) || empty( $type ) ) {
		return;
	}

	// Check that roles are gettable.
	if ( class_exists( 'WP_Roles' ) && ! isset( $wp_roles ) ) {
		$wp_roles = new WP_Roles(); // phpcs:ignore
	}

	// Make sure that $wp_roles is object.
	if ( is_object( $wp_roles ) ) {

		// Make sure parameters are correct type and present.
		if ( ! is_array( $roles ) || empty( $type ) ) {
			return;
		}

		// Create singular and plural types.
		$cap_type  = $type;
		$cap_types = $cap_type . 's';

		// Loop through each role.
		foreach ( $roles as $core_role ) {
			$role = get_role( $core_role );

			// All roles.
			if ( $role ) {

				$role->remove_cap( 'read_' . $cap_type );
				$role->remove_cap( 'edit_' . $cap_type );
				$role->remove_cap( 'delete_' . $cap_type );
				$role->remove_cap( 'edit_' . $cap_types );
				$role->remove_cap( 'publish_' . $cap_types );
				$role->remove_cap( 'delete_published_' . $cap_types );
				$role->remove_cap( 'edit_published_' . $cap_types );
				$role->remove_cap( 'create_' . $cap_types );
				$role->remove_cap( 'upload_files_' . $cap_type );

				// Admin roles.
				if ( in_array( $core_role, [ 'administrator', 'editor' ], true ) ) {
					$role->remove_cap( 'edit_others_' . $cap_types );
					$role->remove_cap( 'read_private_' . $cap_types );
					$role->remove_cap( 'delete_' . $cap_types );
					$role->remove_cap( 'delete_private_' . $cap_types );
					$role->remove_cap( 'delete_others_' . $cap_types );
					$role->remove_cap( 'edit_private_' . $cap_types );
					$role->remove_cap( 'use_others_files_' . $cap_type );
					$role->remove_cap( 'delete_others_files_' . $cap_type );
				}
			}
		}
	}
}
