<?php
/**
 * Nisje Feedback Rest API.
 *
 * @package Nisje
 */

declare( strict_types = 1 );
namespace Nisje\Feedback\Rest;

/**
 * Create Nisje Feedback rest api endpoint.
 */
class Feedback extends \WP_REST_Controller {
	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes(): void {
		$namespace = 'nisje/v1';
		$base      = 'feedback';

		\register_rest_route( $namespace, '/' . $base, [
			[
				'methods'             => \WP_REST_Server::EDITABLE,
				'callback'            => [ $this, 'update_item' ],
				'permission_callback' => [ $this, 'update_item_permissions_check' ],
				'args'                => $this->get_endpoint_args_for_item_schema( false ),
			],
		] );
	}

	/**
	 * Update one item from the collection
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_REST_Response
	 */
	public function update_item( $request ) { // phpcs:ignore
		$id          = $request->get_param( 'id' );
		$type        = $request->get_param( 'type' );
		$description = $request->get_param( 'description' );
		$object_type = $request->get_param( 'object_type' );

		if ( empty( $id ) ) {
			return new \WP_Error( 'invalid_id', __( 'ID is missing', 'nisje' ) );
		}

		$url  = '';
		$text = '';

		if ( 'user-content' === $object_type ) {
			$url  = \home_url( "/activity/$id/" );
			$text = 'Someone has reported the post';
		} elseif ( 'comment' === $object_type ) {
			$url  = \admin_url( "comment.php?action=editcomment&c=$id" );
			$text = 'Someone has reported the comment';
		}

		$type_title = '';

		$report_types = \apply_filters( 'nisje/report_content/email_types', [
			'abusive'      => "It's abusive or harmful",
			'sexual'       => 'Sexual content',
			'suicide'      => 'It expresses intentions of self-harm or suicide',
			'noncompliant' => 'Noncompliance with the Code of Conduct',
			'harassment'   => 'Harassment or discrimination',
			'other'        => 'Other',
		] );

		$type_title = $report_types[ $type ];

		$to      = \get_option( 'nisje_report_email', \get_option( 'admin_email' ) );
		$subject = 'Reported content';
		$body    = $text . ' <a href="' . $url . '">' . $url . '</a>.<br /><br /><b>Type:</b> ' . $type_title . '<br /><b>Description</b>: ' . esc_html( $description );
		$headers = [ 'Content-Type: text/html; charset=UTF-8' ];

		wp_mail( $to, $subject, $body, $headers );

		$response = new \WP_REST_Response( [
			'ok' => true,
		], 200 );

		return $response;
	}

	/**
	 * Check if a given request has access to update a specific item
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_Error|bool
	 */
	public function update_item_permissions_check( $request ) { // phpcs:ignore
		return \is_user_logged_in();
	}

	/**
	 * Prepare the item for create or update operation
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_Error|object $prepared_item
	 */
	protected function prepare_item_for_database( $request ) { // phpcs:ignore
		return [];
	}

	/**
	 * Prepare the item for the REST response
	 *
	 * @param mixed            $items   WordPress representation of the item.
	 * @param \WP_REST_Request $request Request object.
	 * @return mixed
	 */
	public function prepare_item_for_response( $items, $request ) { // phpcs:ignore
		return $items;
	}
}
