<?php
/**
 * Extend Post Types
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

// Types to extend.
$post_types = apply_filters( 'nisje_rest_shared_fields_post_types', [] );

if ( empty( $post_types ) || ! is_array( $post_types ) ) {
	return;
}

// Comment count.
register_rest_field(
	$post_types, 'comment_count', [
		'get_callback' => function ( $object ) {
			$comments = wp_count_comments( $object['id'] );

			return $comments->total_comments;
		},
		'schema'       => [
			'description' => esc_html__( 'Number of comments', 'nisje' ),
			'type'        => 'integer',
			'readonly'    => true,
		],
	]
);

// Comments.
register_rest_field(
	$post_types, 'comments', [
		'get_callback' => function ( $object ) {
			$request = new WP_REST_Request( 'GET', '/wp/v2/comments' );

			$request->set_query_params( [
				'post'     => $object['id'],
				'order'    => 'desc',
				'per_page' => 1,
			] );

			$response = rest_do_request( $request );
			$server   = rest_get_server();
			$data     = $server->response_to_data( $response, false );

			return array_reverse( $data, false );
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Comments object', 'nisje' ),
			'type'        => 'object',
			'readonly'    => true,
		],
	]
);

register_rest_field(
	$post_types, 'author_name', [
		'get_callback' => function( $object ) {
			$retval = '';

			$post = get_post( $object['id'] );
			if ( $post instanceof \WP_Post ) {
				$retval = bp_core_get_user_displayname( $post->post_author );
			}

			return $retval;
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Author Name', 'nisje' ),
			'type'        => 'string',
			'readonly'    => true,
		],
	]
);

// Image data.
register_rest_field(
	$post_types, 'featuredImage', [
		'get_callback' => function( $object ) {
			if ( has_post_thumbnail( $object['id'] ) ) {
				$attachment_id = get_post_thumbnail_id( $object['id'] );
				$image         = wp_get_attachment_image_src( $attachment_id, 'medium_large', false );

				if ( $image ) {
					list( $src, $width, $height ) = $image;

					$attr = [
						'src' => $src,
						'alt' => trim( strip_tags( get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) ) ), // phpcs:ignore
					];

					// Generate 'srcset' and 'sizes' if not already present.
					if ( empty( $attr['srcSet'] ) ) {
						$image_meta = wp_get_attachment_metadata( $attachment_id );

						if ( is_array( $image_meta ) ) {
							$size_array = [ absint( $width ), absint( $height ) ];
							$srcset     = wp_calculate_image_srcset( $size_array, $src, $image_meta, $attachment_id );
							$sizes      = wp_calculate_image_sizes( $size_array, $src, $image_meta, $attachment_id );

							if ( $srcset && ( $sizes || ! empty( $attr['sizes'] ) ) ) {
								$attr['srcSet'] = $srcset;

								if ( empty( $attr['sizes'] ) ) {
									$attr['sizes'] = $sizes;
								}
							}
						}
					}

					$attr = array_map( 'esc_attr', $attr );

					return $attr;
				} else {
					return [];
				}
			}

			return [];
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Image data', 'nisje' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'string',
			],
			'readonly'    => true,
		],
	]
);

// Reactions.
register_rest_field(
	$post_types, 'reactions', [
		'get_callback' => function ( $object ) {
			return nisje_get_user_reactions( $object['id'], 'post' );
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Reactions', 'nisje' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'mixed',
			],
			'readonly'    => true,
		],
	]
);

// Attachments.
register_rest_field(
	$post_types, 'attachments', [
		'get_callback'    => 'nisje_get_post_attachments',
		'update_callback' => 'nisje_update_post_attachments',
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'IDs of attachments', 'nisje' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => false,
		],
	]
);
