<?php
/**
 * Setup Groups Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Groups Component Class
 */
class Groups extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		parent::start(
			'groups',
			__( 'Groups', 'nisje' )
		);
	}

	/**
	 * Setup actions
	 */
	public function setup_filters() {
		\add_action( 'bp_groups_admin_meta_boxes', [ $this, 'setup_meta_boxes' ] );
		\add_action( 'bp_group_admin_edit_after', [ $this, 'save_meta_boxes' ] );
		\add_action( 'rest_api_init', [ $this, 'setup_rest_api' ] );
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		nisje_include( 'includes/components/groups/rest-handlers/class-groups-controller.php' );
		$controller = new \Dekode\Nisje\Components\Rest\Groups_Controller();
		$controller->register_routes();

		nisje_include( 'includes/components/groups/rest-handlers/class-groups-members-controller.php' );
		$controller = new \Dekode\Nisje\Components\Rest\Groups_Members_Controller();
		$controller->register_routes();

		nisje_include( 'includes/components/groups/rest-handlers/class-groups-memberships-controller.php' );
		$controller = new \Dekode\Nisje\Components\Rest\Groups_Memberships_Controller();
		$controller->register_routes();

		nisje_include( 'includes/components/groups/rest-handlers/class-groups-invites-controller.php' );
		$controller = new \Dekode\Nisje\Components\Rest\Groups_Invites_Controller();
		$controller->register_routes();
	}

	/**
	 * Setup group meta boxes
	 */
	public function setup_meta_boxes() {
		add_meta_box(
			'nisje_group_allowed_posting',
			_x( 'Settings', 'group admin edit screen', 'nisje' ),
			[ $this, 'render_meta_box' ],
			get_current_screen()->id,
			'side',
			'core'
		);
	}

	/**
	 * Render group meta box
	 *
	 * @param object $item Group object.
	 */
	public function render_meta_box( $item ) {
		$access = \groups_get_groupmeta( $item->id, 'nisje_group_allowed_posting' ) ?: 'members';
		?>
		<div class="bp-groups-settings-section" id="bp-groups-settings-section-status">
			<fieldset>
				<legend><?php esc_html_e( 'Who is allowed to add posts?', 'nisje' ); ?></legend>

				<label for="nisje-group-allowed-members"><input type="radio" name="group-allowed-posting" id="nisje-group-allowed-members" value="members" <?php checked( $access, 'members' ); ?> /><?php esc_html_e( 'All group members', 'nisje' ); ?></label>
				<label for="nisje-group-allowed-mods"><input type="radio" name="group-allowed-posting" id="nisje-group-allowed-mods" value="mods" <?php checked( $access, 'mods' ); ?> /><?php esc_html_e( 'Group moderators and admins', 'nisje' ); ?></label>
				<label for="nisje-group-allowed-admins"><input type="radio" name="group-allowed-posting" id="nisje-group-allowed-admins" value="admins" <?php checked( $access, 'admins' ); ?> /><?php esc_html_e( 'Group admins', 'nisje' ); ?></label>
			</fieldset>
		</div>
		<?php
	}

	/**
	 * Save group meta boxes
	 *
	 * @param int $group_id Group ID.
	 */
	public function save_meta_boxes( $group_id ) {
		if ( ! isset( $_POST['group-allowed-posting'] ) ) { // phpcs:ignore
			return;
		}

		$access = sanitize_key( $_POST['group-allowed-posting'] ); // phpcs:ignore
		\groups_update_groupmeta( $group_id, 'nisje_group_allowed_posting', $access );
	}

	/**
	 * Setup REST API
	 */
	public function setup_rest_api() {
		\register_rest_field( 'bp_groups', 'post_access', [
			'get_callback' => function( $object ) {
				return \groups_get_groupmeta( $object['id'], 'nisje_group_allowed_posting' ) ?: 'members';
			},
			'schema'       => [
				'context'     => [ 'view', 'edit' ],
				'description' => \esc_html__( 'Who can post', 'nisje' ),
				'type'        => 'string',
				'readonly'    => true,
			],
		] );
	}
}
