<?php
/**
 * Base Group Post Endpoints
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components\Rest;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Base Group Post Rest Class
 */
class Search_Controller extends \WP_REST_Controller {
	/**
	 * Search handlers used by the controller.
	 *
	 * @var array
	 */
	protected $search_handlers = [];

	/**
	 * Constructor.
	 *
	 * @param array $search_handlers List of search handlers to use in the controller.
	 */
	public function __construct( array $search_handlers ) {
		$this->namespace = 'nisje/v1';
		$this->rest_base = 'search';

		foreach ( $search_handlers as $search_handler ) {
			$this->search_handlers[ $search_handler->get_type() ] = $search_handler;
		}
	}

	/**
	 * Registers the routes for the objects of the controller.
	 *
	 * @see register_rest_route()
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			[
				[
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => [ $this, 'get_items' ],
					'permission_callback' => [ $this, 'get_items_permission_check' ],
					'args'                => $this->get_collection_params(),
				],
				'schema' => [ $this, 'get_public_item_schema' ],
			]
		);
	}

	/**
	 * Checks if a given request has access to search content.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 * @return true|\WP_Error True if the request has search access, \WP_Error object otherwise.
	 */
	public function get_items_permission_check( $request ) {
		return true;
	}

	/**
	 * Retrieves a collection of search results.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|\WP_Error Response object on success, or \WP_Error object on failure.
	 */
	public function get_items( $request ) {
		$responses = [];

		foreach ( $this->search_handlers as $handler ) {
			if ( \is_wp_error( $handler ) ) {
				return $handler;
			}

			$result = $handler->search_items( $request );

			if ( ! isset( $result[ \WP_REST_Search_Handler::RESULT_IDS ] ) || ! is_array( $result[ \WP_REST_Search_Handler::RESULT_IDS ] ) || ! isset( $result[ \WP_REST_Search_Handler::RESULT_TOTAL ] ) ) {
				return new \WP_Error(
					'rest_search_handler_error',
					__( 'Internal search handler error.', 'nisje' ),
					[ 'status' => 500 ]
				);
			}

			$ids = $result[ \WP_REST_Search_Handler::RESULT_IDS ];

			$results = [
				'items' => [],
			];

			foreach ( $ids as $id ) {
				$data               = $this->prepare_for_response( $id, $request, $handler );
				$results['items'][] = $this->prepare_response_for_collection( $data );
			}

			$total     = (int) $result[ \WP_REST_Search_Handler::RESULT_TOTAL ];
			$page      = (int) $request['page'];
			$per_page  = (int) $request['per_page'];
			$max_pages = ceil( $total / $per_page );

			$results['total']      = $total;
			$results['totalPages'] = $max_pages;

			$responses[ $handler->get_type() ] = $results;
		}

		return rest_ensure_response( $responses );
	}

	/**
	 * Prepares a single search result for response.
	 *
	 * @param int|string       $id      ID of the item to prepare.
	 * @param \WP_REST_Request $request Request object.
	 * @param class            $handler Search handler.
	 * @return WP_REST_Response Response object.
	 */
	public function prepare_for_response( $id, $request, $handler ) {
		if ( \is_wp_error( $handler ) ) {
			return new WP_REST_Response();
		}

		$fields = $this->get_fields_for_response( $request );

		$data = $handler->prepare_item( $id, $fields );
		$data = $this->add_additional_fields_to_object( $data, $request );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->filter_response_by_context( $data, $context );

		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * Retrieves the item schema, conforming to JSON Schema.
	 *
	 * @return array Item schema data.
	 */
	public function get_item_schema() {
		if ( $this->schema ) {
			return $this->add_additional_fields_schema( $this->schema );
		}

		$schema = [
			'$schema' => 'http://json-schema.org/draft-04/schema#',
			'title'   => 'search-result',
			'type'    => 'object',
		];

		$this->schema = $schema;

		return $this->add_additional_fields_schema( $this->schema );
	}

	/**
	 * Retrieves the query params for the search results collection.
	 *
	 * @return array Collection parameters.
	 */
	public function get_collection_params() {
		$query_params = parent::get_collection_params();

		$query_params['context']['default'] = 'view';

		return $query_params;
	}
}
