<?php
/**
 * Setup Profile Layouts Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Profile Layouts Component Class
 */
class Profile_Layouts extends Component {
	/**
	 * Key
	 *
	 * @var string
	 */
	private $key;

	/**
	 * Constructor
	 */
	public function __construct() {
		parent::start(
			'profile-layouts',
			__( 'Profile Layouts', 'nisje' )
		);
	}

	/**
	 * Includes
	 */
	public function includes() {
		nisje_include( 'includes/components/profile-layouts/helper-functions.php' );
		nisje_include( 'includes/components/profile-layouts/classes/class-nisje-profile-layout.php' );
	}

	/**
	 * Setup globals
	 */
	public function setup_globals() {
		$this->key = 'di_profile_layouts';

		$settings = get_option( $this->key . '_settings' );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			foreach ( $settings as $key => $value ) {
				$this->settings[ $key ] = $value;
			}
		}

		$namespace_key = wp_cache_get( 'di_profile_layouts_namespace_key' );
		if ( false === $namespace_key ) {
			wp_cache_set( 'di_profile_layouts_namespace_key', 1 );
			$namespace_key = 1;
		}

		$defaults = [
			'key'               => $this->key,
			'name'              => esc_html__( 'Profile Layouts', 'nisje' ),
			'settings_menu_key' => 'di-profile-layouts',
			'cache_key'         => $namespace_key,
		];

		$this->settings = array_merge( $defaults, (array) $this->settings );

		$this->register_default_profile_layouts();
	}

	/**
	 * Setup Filters
	 */
	public function setup_filters() {
		add_filter( 'nisje_custom_profile_layouts', [ $this, 'register_profile_layouts' ] );
		add_filter( 'nisje_custom_profile_layout_value_spacer', [ $this, 'register_profile_layout_value_spacer' ], 10, 2 );
		add_filter( 'nisje_custom_profile_layout_value_title', [ $this, 'register_profile_layout_value_title' ], 10, 2 );

		if ( is_admin() ) {
			add_action( 'nisje_settings_add_option_page', [ $this, 'add_menu' ] );
			add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_groups' ] );
			add_action( 'nisje_settings_populate_options_from_acf', [ $this, 'populate_acf_settings' ] );
		}

		add_filter( 'nisje_custom_profile_layouts', [ $this, 'register_profile_layout' ] );
		add_filter( 'nisje_custom_profile_layout_value_member_type', [ $this, 'register_profile_layout_value' ], 10, 2 );

		// @TODO Flush profile field cache.
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		nisje_include( 'includes/components/profile-layouts/rest-handlers/extend-members.php' );
		nisje_include( 'includes/components/profile-layouts/rest-handlers/extend-user.php' );
		nisje_include( 'includes/components/profile-layouts/rest-handlers/extend-user-content.php' );

		if ( bp_is_active( 'groups' ) ) {
			nisje_include( 'includes/components/profile-layouts/rest-handlers/extend-group-members.php' );
		}
	}

	/**
	 * Register defaults.
	 */
	public function register_default_profile_layouts() {
		nisje_register_profile_layout( 'di_key_member_profile_header_fields', 'di_member_profile_header_fields', esc_html__( 'Profile Member header fields', 'nisje' ) );
		nisje_register_profile_layout( 'di_key_member_profile_about_fields', 'di_member_profile_about_fields', esc_html__( 'Profile Member about page fields', 'nisje' ) );
		nisje_register_profile_layout( 'di_key_member_listing_fields', 'di_member_listing_fields', esc_html__( 'Member listing', 'nisje' ) );
		nisje_register_profile_layout( 'di_key_member_listing_header_fields', 'di_member_listing_header_fields', esc_html__( 'Member listing header fields', 'nisje' ) );
	}

	/**
	 * Register core profile layouts.
	 *
	 * @param array $layouts Registered profile layouts.
	 * @return array $layouts Populated profile layouts.
	 */
	public function register_profile_layouts( $layouts ) {
		$layouts['spacer'] = esc_html__( 'Spacer', 'nisje' );
		$layouts['title']  = esc_html__( 'Title', 'nisje' );

		return $layouts;
	}

	/**
	 * Profile layout spacer value.
	 *
	 * @param array $field   Profile field.
	 * @param array $user_id User ID.
	 *
	 * @return array Array with profile values.
	 */
	public function register_profile_layout_value_spacer( $field, $user_id ) {
		return [
			'name'  => esc_html__( 'Spacer', 'nisje' ),
			'value' => 'spacer',
			'type'  => $field['field'],
		];
	}

	/**
	 * Profile layout title value.
	 *
	 * @param array $field   Profile field.
	 * @param array $user_id User ID.
	 *
	 * @return array Array with profile values.
	 */
	public function register_profile_layout_value_title( $field, $user_id ) {
		return [
			'name'  => esc_html__( 'Title', 'nisje' ),
			'value' => '',
			'type'  => $field['field'],
		];
	}

	/**
	 * Setup the admin page.
	 */
	public function add_menu() {
		acf_add_options_sub_page( [
			'page_title'  => $this->settings['name'],
			'menu_title'  => $this->settings['name'],
			'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
			'menu_slug'   => nisje_get_setting( 'settings_menu_key', 'profile_layouts' ),
			'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
		] );
	}

	/**
	 * Include acf field groups.
	 */
	public function register_field_groups() {
		nisje_include( 'includes/components/profile-layouts/acf/field-groups/acf-field-group-options-profile-layouts.php' );
	}

	/**
	 * Populate ACF Options General
	 *
	 * @param WP_Screen $screen Screen object.
	 */
	public function populate_acf_settings( $screen ) {
		if ( false !== strpos( $screen->id, nisje_get_setting( 'settings_menu_key', 'profile_layouts' ) ) ) {
			$profile_layouts = [];

			$layouts = nisje_get_profile_layouts();

			if ( ! empty( $layouts ) ) {
				foreach ( $layouts as $layout ) {
					if ( isset( $layout->name ) ) {
						$options = get_field( $layout->name, 'option' );
						foreach ( (array) $options as $option ) {
							if ( 'di_profile_layout' === $option['acf_fc_layout'] ) {
								$profile_layouts['layouts'][ $layout->name ][] = [
									'type'    => $option['acf_fc_layout'],
									'field'   => $option['di_profile_field'][0],
									'format'  => $option['di_format_field'],
									'prepend' => $option['di_prepend_field'],
								];
							} elseif ( 'di_wp_layout' === $option['acf_fc_layout'] ) {
								$profile_layouts['layouts'][ $layout->name ][] = [
									'type'    => $option['acf_fc_layout'],
									'field'   => $option['di_wp_field'],
									'format'  => $option['di_format_field'],
									'prepend' => $option['di_prepend_field'],
								];
							} elseif ( 'di_custom_layout' === $option['acf_fc_layout'] ) {
								$profile_layouts['layouts'][ $layout->name ][] = [
									'type'    => $option['acf_fc_layout'],
									'field'   => $option['di_custom_field'],
									'format'  => $option['di_format_field'],
									'prepend' => $option['di_prepend_field'],
								];
							}
						}
					}
				}
			}

			update_option( $this->key . '_settings', $profile_layouts );

			$cache_key = nisje_get_setting( 'cache_key', 'profile_layouts' );
			wp_cache_incr( $cache_key );
		}
	}

	/**
	 * Register profile layout.
	 *
	 * @param array $layouts Custom layouts.
	 * @return array $layouts Optimized layouts.
	 */
	public function register_profile_layout( $layouts ) {
		$layouts['member_type'] = esc_html__( 'Member Type', 'nisje' );

		return $layouts;
	}

	/**
	 * Register profile layout value.
	 *
	 * @param array $field   Field data.
	 * @param int   $user_id The user id.
	 *
	 * @return array Value.
	 */
	public function register_profile_layout_value( $field, $user_id ) {
		$retval = bp_get_member_type( $user_id, true );
		if ( $retval ) {
			$type_object = bp_get_member_type_object( $retval );
			if ( $type_object && isset( $type_object->labels['name'] ) ) {
				$retval = $type_object->labels['name'];
			}

			return [
				'name'  => esc_html__( 'Member Type', 'nisje' ),
				'value' => $retval,
				'type'  => 'member_type',
			];
		}
	}
}
