<?php
/**
 * Digest Commands.
 *
 * @package Nisje
 */

declare( strict_types = 1 );
namespace Nisje\CLI\Command;

use WP_CLI;

/**
 * Digest class.
 */
class Digest {
	/**
	 * Send daily digests.
	 *
	 * ## EXAMPLE
	 *
	 *     $ wp nisje digest daily
	 */
	public function daily(): void {
		$start_date = '-2 days midnight';
		$end_date   = '-1 days midnight';

		$groups  = [];
		$content = [
			'posts' => [],
		];

		$posts_query = new \WP_Query( [
			'post_type'           => 'nisje-user-content',
			'post_status'         => 'publish',
			'posts_per_page'      => -1,
			'ignore_sticky_posts' => true,
			'date_query'          => [
				'after'  => $start_date,
				'before' => $end_date,
			],
		] );

		if ( $posts_query->have_posts() ) {
			while ( $posts_query->have_posts() ) {
				$posts_query->the_post();
				$id = \get_the_ID();

				$content['posts'][] = [
					'id'             => $id,
					'group_id'       => \wp_get_post_parent_id( $id ),
					'content'        => \wp_strip_all_tags( \get_the_content() ),
					'author'         => \get_the_author(),
					'author_url'     => \home_url( '/members/' . \get_the_author_meta( 'ID' ) ),
					'comments_count' => \get_comments_number(),
					'tags'           => \get_the_terms( $id, 'nisje-tags' ),
					'images'         => \get_post_meta( $id, 'nisje-images', true ),
				];
			}
		} else {
			\WP_CLI::line( 'No new posts found.' );
			return;
		}

		$group_ids = array_values( array_unique( \wp_list_pluck( $content['posts'], 'group_id' ) ) );

		// Remove 0 from group ids.
		$group_zero = array_search( 0, $group_ids, true );
		if ( false !== $group_zero ) {
			unset( $group_ids[ $group_zero ] );
		}

		$digest_groups = \groups_get_groups( [
			'include'     => $group_ids,
			'show_hidden' => true,
		] );

		foreach ( $digest_groups['groups'] as $group ) {
			$groups[ $group->id ] = [
				'name'   => $group->name,
				'avatar' => \bp_core_fetch_avatar( [
					'item_id' => $group->id,
					'object'  => 'group',
					'type'    => 'full',
					'html'    => false,
				] ),
			];
		}

		/**
		 * Users
		 */
		$users = get_users( [
			'fields' => [ 'ID', 'display_name', 'user_email' ],
		] );

		foreach ( $users as $user ) {
			\WP_CLI::line( 'User: Creating digest for ' . $user->display_name . ' (' . $user->ID . ')' );

			$args = [
				'title'    => \get_bloginfo( 'name' ),
				'logo'     => '',
				'date'     => \gmdate( 'j. F Y' ),
				'email'    => $user->user_email,
				'home_url' => \home_url( '/' ),
			];

			$group_ids = \groups_get_user_groups( $user->ID );

			ob_start();
			include __DIR__ . '/emails/header.php';

			foreach ( $content['posts'] as $user_post ) {
				if ( 0 === $user_post['group_id'] || in_array( $user_post['group_id'], $group_ids['groups'] ) ) { // phpcs:ignore
					include __DIR__ . '/emails/user-post.php';
				}
			}

			include __DIR__ . '/emails/footer.php';
			$output = ob_get_contents();
			ob_end_clean();

			if ( ! empty( $user->user_email ) ) {
				wp_mail( $user->user_email, 'Daily digest', $output, [ 'Content-Type: text/html; charset=UTF-8' ] );
			}
		}
	}
}
