<?php
/**
 * Reaction Helper Functions
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Get reaction type.
 *
 * @param string $reaction_type_id Reaction type ID.
 *
 * @return array|bool
 */
function nisje_get_reaction_type( $reaction_type_id ) {
	$reaction_types = nisje_get_setting( 'reaction_types', 'reaction' );

	if ( isset( $reaction_types[ $reaction_type_id ] ) ) {
		return $reaction_types[ $reaction_type_id ];
	} else {
		return false;
	}
}

/**
 * Get single reaction
 *
 * @param int    $object_id        Object ID.
 * @param int    $user_id          User ID.
 * @param string $object_type      Object Type.
 * @param string $reaction_type_id Reaction type ID.
 *
 * @return mixed
 */
function nisje_get_reaction( $object_id, $user_id, $object_type, $reaction_type_id = 0 ) {
	$reaction = new Nisje_Reaction( $object_id, $user_id, $object_type, $reaction_type_id );

	if ( ! $reaction->id ) {
		return false;
	}

	return $reaction;
}


/**
 * Reacting
 *
 * @param int    $object_id        Object ID.
 * @param int    $user_id          User ID.
 * @param string $object_type      Object Type.
 * @param string $reaction_type_id Reaction type ID.
 *
 * @return bool
 */
function nisje_add_reaction( $object_id, $user_id, $object_type, $reaction_type_id = 0 ) {
	$reaction_type = nisje_get_reaction_type( $reaction_type_id );
	if ( ! $reaction_type ) {
		return false;
	}

	if ( ! in_array( $object_type, [ 'post', 'comment', 'activity' ], true ) ) {
		return false;
	}

	$reaction = new Nisje_Reaction( $object_id, $user_id, $object_type, $reaction_type_id );

	if ( $reaction->id ) {
		return false;
	}

	if ( ! $reaction->save() ) {
		return false;
	}

	do_action( 'nisje_add_reaction', $object_id, $user_id, $object_type, $reaction_type_id );

	return true;
}

/**
 * Remove reaction.
 *
 * @param int    $object_id        Object ID.
 * @param int    $user_id          User ID.
 * @param string $object_type      Object Type.
 * @param string $reaction_type_id Reaction type ID.
 *
 * @return bool
 */
function nisje_remove_reaction( $object_id, $user_id, $object_type, $reaction_type_id ) {
	$reaction = new Nisje_Reaction( $object_id, $user_id, $object_type, $reaction_type_id );

	if ( ! $reaction->delete() ) {
		return false;
	}

	do_action( 'nisje_remove_reaction', $object_id, $user_id, $object_type, $reaction_type_id );

	return true;
}

/**
 * Get users who reacted to an object.
 *
 * @param int    $object_id   Object ID.
 * @param string $object_type Object Type.
 * @return array
 */
function nisje_get_user_reactions( $object_id, $object_type ) {
	return Nisje_Reaction::get_reactions_connected_to_object( $object_id, $object_type );
}

/**
 * Check if user has reacted in reactions.
 *
 * @param int   $user_id   User ID.
 * @param array $reactions Reactions.
 * @return bool
 */
function nisje_user_has_reacted( $user_id, $reactions ) {
	foreach ( $reactions as $reaction ) {
		if ( $user_id === $reaction->user_id ) {
			return true;
		}
	}

	return false;
}
