<?php
/**
 * Nisje Digest Rest API.
 *
 * @package Nisje
 */

declare( strict_types = 1 );
namespace Nisje\Extension\Digest\Rest;

use function Nisje\Utils\encrypt_decrypt;

/**
 * Nisje Digest Members endpoints.
 */
class Members extends \WP_REST_Users_Controller {
	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace = 'nisje/v1';
		$this->rest_base = 'digest-members';
	}

	/**
	 * Retrieve users.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response
	 */
	public function get_items( $request ) {
		$args = [
			'per_page' => $request->get_param( 'per_page' ),
			'page'     => $request->get_param( 'page' ),
		];

		// Actually, query it.
		$member_query = new \BP_User_Query( $args );
		$members      = array_values( $member_query->results );

		$retval = [];
		foreach ( $members as $member ) {
			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $member, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = \bp_rest_response_add_total_headers( $response, $member_query->total_users, $args['per_page'] );

		return $response;
	}

	/**
	 * Checks if a given request has access to get all users.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 */
	public function get_items_permissions_check( $request ) {
		return \is_user_logged_in() && \current_user_can( 'manage_options' );
	}

	/**
	 * Prepares a single user output for response.
	 *
	 * @param WP_User         $user    User object.
	 * @param WP_REST_Request $request Full details about the request.
	 */
	public function prepare_item_for_response( $user, $request ) {
		$data     = $this->user_data( $user );
		$response = \rest_ensure_response( $data );

		return $response;
	}

	/**
	 * Method to facilitate fetching of user data.
	 *
	 * @param WP_User $user User object.
	 * @return array
	 */
	public function user_data( $user ) {
		$group = \groups_get_user_groups( $user->ID );
		$data  = [
			'id'          => (int) $user->ID,
			'name'        => $user->display_name,
			'email'       => $user->user_email,
			'groups'      => $group['groups'],
			'schedule'    => \get_user_meta( $user->ID, 'nisje_digest_schedule', true ) ?: 'never',
			'unsubscribe' => encrypt_decrypt( 'encrypt', \wp_json_encode( [
				'email' => $user->user_email,
				'id'    => (string) $user->ID,
			] ) ),
		];

		return $data;
	}

	/**
	 * Get the members schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		if ( is_null( $this->schema ) ) {
			$schema = [
				'$schema'    => 'http://json-schema.org/draft-04/schema#',
				'title'      => 'digest_members',
				'type'       => 'object',
				'properties' => [
					'id'       => [
						'description' => 'A unique numeric ID for the Member.',
						'type'        => 'integer',
						'context'     => [ 'view' ],
						'readonly'    => true,
					],
					'name'     => [
						'description' => 'Display name for the member.',
						'type'        => 'string',
						'context'     => [ 'view' ],
						'readonly'    => true,
					],
					'email'    => [
						'description' => 'Email for the member.',
						'type'        => 'string',
						'context'     => [ 'view' ],
						'readonly'    => true,
					],
					'groups'   => [
						'description' => 'Groups.',
						'type'        => 'array',
						'context'     => [ 'view' ],
						'readonly'    => true,
					],
					'schedule' => [
						'description' => 'Digest schedule.',
						'type'        => 'string',
						'context'     => [ 'view' ],
						'readonly'    => true,
					],
				],
			];

			// Cache current schema here.
			$this->schema = $schema;
		}

		return $this->add_additional_fields_schema( $this->schema );
	}

	/**
	 * Get the query params for collections.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = array_intersect_key(
			parent::get_collection_params(),
			[
				'page'     => true,
				'per_page' => true,
			]
		);

		return $params;
	}
}
