<?php
/**
 * Nisje BuddyPress Archive Group
 *
 * @package Nisje
 */

namespace Nisje\BuddyPress\ArchiveGroup;

const META_KEY = '_archived_group';

/**
 * Check if archive filter is active.
 */
function is_archive_filter_active(): bool {
	return isset( $_GET['archived'] ) && '1' === $_GET['archived']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
}

/**
 * Add filter to archived groups.
 *
 * @param string $url_base Current URL base for view.
 * @param string $view     Current view being displayed.
 */
function add_group_archive_filter( $url_base, $view ) {
	$classname = is_archive_filter_active() ? 'current' : '';

	printf( '| <li><a href="%s" class="%s">%s</a></li>',
		\esc_url( \add_query_arg( 'archived', '1', $url_base ) ),
		\esc_attr( $classname ),
		\esc_html__( 'Archived', 'nisje' )
	);
}
\add_action( 'bp_groups_list_table_get_views', __NAMESPACE__ . '\\add_group_archive_filter', 10, 2 );

/**
 * Add link to archive group.
 *
 * @param array $actions Array of actions to be displayed for the column content.
 * @param array $item    The current group item in the loop.
 */
function add_group_archive_link( array $actions, array $item ): array {
	$nonce_key = 'bp-groups-archive';
	$base_url  = \admin_url( 'admin.php?page=bp-groups&gid=' . $item['id'] );

	if ( is_archive_filter_active() ) {
		$archive_url          = \wp_nonce_url( $base_url . '&action=unarchive', $nonce_key );
		$actions['unarchive'] = sprintf( '<a href="%s">%s</a>', esc_url( $archive_url ), __( 'Unarchive', 'nisje' ) );
	} else {
		$archive_url        = \wp_nonce_url( $base_url . '&archived=1&action=archive', $nonce_key );
		$actions['archive'] = sprintf( '<a href="%s">%s</a>', esc_url( $archive_url ), __( 'Archive', 'nisje' ) );
	}

	return $actions;
}
\add_action( 'bp_groups_admin_comment_row_actions', __NAMESPACE__ . '\\add_group_archive_link', 10, 2 );

/**
 * Archive or unarchive a group.
 *
 * @param string $do_action Action to perform.
 */
function do_archive( string $do_action ): void {
	if ( ! empty( $_GET['gid'] ) ) {
		if ( 'archive' === $do_action ) {
			check_admin_referer( 'bp-groups-archive' );
			$group_id = (int) $_GET['gid'];
			archive_group( $group_id, true );
			bp_core_redirect( \admin_url( 'admin.php?page=bp-groups&archived=1&archived_message=1&gid=' . $group_id ) );
		} elseif ( 'unarchive' === $do_action ) {
			check_admin_referer( 'bp-groups-archive' );
			$group_id = (int) $_GET['gid'];
			archive_group( $group_id, false );
			bp_core_redirect( \admin_url( 'admin.php?page=bp-groups&archived_message=2&gid=' . $group_id ) );
		}
	}
}
\add_action( 'bp_groups_admin_load', __NAMESPACE__ . '\\do_archive' );

/**
 * Archive or unarchive a group message.
 *
 * @param array $messages Array of messages to be displayed.
 */
function add_archived_message( array $messages ): void {
	// If the user has just made a change to a group, build status messages.
	if ( ! empty( $_GET['archived_message'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$action = (int) $_GET['archived_message']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( 1 === $action ) {
			?>
			<div id="moderated" class="updated notice is-dismissible"><p><?php \esc_html_e( 'Group has been archived.', 'nisje' ); ?></p></div>
			<?php
		} elseif ( 2 === $action ) {
			?>
			<div id="moderated" class="updated notice is-dismissible"><p><?php \esc_html_e( 'Group has been unarchived.', 'nisje' ); ?></p></div>
			<?php
		}
	}
}
\add_action( 'bp_groups_admin_index', __NAMESPACE__ . '\\add_archived_message' );

/**
 * Filters the BP_Groups_Group::get() SQL queries
 *
 * There's no great way to do this in BP. Parse the SQL query string, so it
 * maintain maximum compatibility with other plugins that might manipulate the
 * query.
 *
 * @param string $sql SQL query string.
 */
function filter_sql( string $sql ): string {
	global $bp, $wpdb;

	if ( \is_admin() && isset( $_GET['archived'] ) && '1' === $_GET['archived'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$include_groups = $wpdb->get_col( $wpdb->prepare( "SELECT group_id FROM {$bp->groups->table_name_groupmeta} WHERE meta_key = %s AND meta_value = %s", META_KEY, 'archived' ) ); // phpcs:ignore
		$in_sql         = ' g.id IN (' . implode( ',', $include_groups ) . ') AND ';
		$sql_a          = explode( 'WHERE', $sql );
		$sql            = $sql_a[0] . 'WHERE' . $in_sql . $sql_a[1];
	} elseif ( \strpos( $sql, 'WHERE' ) !== false ) {
		$sql_a     = explode( 'WHERE', $sql );
		$sub_query = $wpdb->prepare( "SELECT group_id FROM {$bp->groups->table_name_groupmeta} gm_ag WHERE g.id = gm_ag.group_id AND gm_ag.meta_key = %s", [ META_KEY ] ); // phpcs:ignore
		$sql       = $sql_a[0] . 'WHERE NOT EXISTS ( ' . $sub_query . ' ) AND ' . $sql_a[1];
	}

	return $sql;
}
\add_filter( 'bp_groups_get_paged_groups_sql', __NAMESPACE__ . '\\filter_sql' );
\add_filter( 'bp_groups_get_total_groups_sql', __NAMESPACE__ . '\\filter_sql' );

/**
 * Archive a group
 *
 * @param int  $group_id Group ID.
 * @param bool $archive  Whether to archive or unarchive the group.
 */
function archive_group( int $group_id, bool $archive = true ): bool {
	if ( $archive ) {
		return \groups_update_groupmeta( $group_id, META_KEY, 'archived' );
	}

	return \groups_delete_groupmeta( $group_id, META_KEY );
}
