<?php
/**
 * Group connection utility functions.
 *
 * @package Nisje
 */

namespace Nisje\GroupConnection\Utils;

use const Nisje\Constants\GROUP_POST_CAP;

/**
 * Add connection field to groups.
 *
 * @param string $post_type_name The post type.
 * @param string $meta_key       Meta key.
 */
function add_group_connection_meta( string $post_type_name, string $meta_key ): void {
	// Add meta field to editor.
	\add_action( 'enqueue_block_editor_assets', function () use ( $post_type_name, $meta_key ) {
		global $post_type;
		$asset_file = NISJE_CORE_DIR_PATH . 'utils/group-connection/build/editor.asset.php';

		if ( ! $post_type || $post_type_name !== $post_type || ! \file_exists( $asset_file ) ) {
			return;
		}

		$groups      = [];
		$user_groups = \groups_get_user_groups( \get_current_user_id() );

		foreach ( $user_groups['groups'] as $group_id ) {
			$is_connection_enabled = (bool) \groups_get_groupmeta( $group_id, $meta_key, true );

			if ( $is_connection_enabled ) {
				$groups[] = [
					'label' => \groups_get_group( $group_id )->name,
					'value' => (int) $group_id,
				];
			}
		}

		$asset        = require_once $asset_file;
		$dependencies = isset( $asset['dependencies'] ) ? $asset['dependencies'] : [];
		$version      = isset( $asset['version'] ) ? $asset['version'] : filemtime( $path );

		\wp_enqueue_script(
			'nisje-group-connection-meta-field',
			NISJE_CORE_URL . 'utils/group-connection/build/editor.js',
			$dependencies,
			$version,
			true
		);

		\wp_add_inline_script(
			'nisje-group-connection-meta-field',
			sprintf(
				'var nisjeGroupConnections = %s;',
				\wp_json_encode( $groups )
			),
			'before'
		);
	}, 20 );
}

/**
 * Get default post type arguments.
 */
function get_default_post_type_args(): array {
	return [
		'public'                => true,
		'publicly_queryable'    => true,
		'show_in_admin_bar'     => false,
		'show_in_nav_menus'     => false,
		'can_export'            => false,
		'query_var'             => false,
		'capability_type'       => GROUP_POST_CAP,
		'capabilities'          => [
			'create_posts'     => 'create_' . GROUP_POST_CAP . 's',
			'delete_posts'     => 'delete_' . GROUP_POST_CAP . 's',
			'upload_files_cpt' => 'upload_files_' . GROUP_POST_CAP,
		],
		'map_meta_cap'          => true,
		'menu_icon'             => 'dashicons-media-text',
		'show_in_rest'          => true,
		'rest_controller_class' => '\\Dekode\\Nisje\\Components\\Rest\\Base_Group_Post_Controller',
		'show_in_graphql'       => true,
	];
}

/**
 * Add frontend settings
 *
 * @param string $key Settings key.
 */
function add_frontend_capability( string $key ): void {
	\add_filter( 'nisje_frontend_options', function( array $settings ) use ( $key ): array {
		if ( \is_user_logged_in() ) {
			$user_id = \get_current_user_id();
			$user    = \get_userdata( $user_id );

			$settings['userCapabilities']['edit'][ $key ]   = $user->allcaps[ 'edit_' . GROUP_POST_CAP ] ?? false;
			$settings['userCapabilities']['create'][ $key ] = $user->allcaps[ 'create_' . GROUP_POST_CAP . 's' ] ?? false;
		}

		return $settings;
	} );
}

/**
 * Extend WPGraphQL schema with edit cap
 *
 * @param string $graphql_name GraphQL schema name.
 */
function add_can_edit_to_graphql_schema( string $graphql_name ): void {
	\add_action( 'graphql_register_types', function() use ( $graphql_name ): void {
		\register_graphql_field( $graphql_name, 'canEdit', [
			'type'    => 'boolean',
			'resolve' => function( $source ) {
				$user_id = \get_current_user_id();

				if ( ! $user_id ) {
					return false;
				}

				return \groups_is_user_member( $user_id, $source->parentDatabaseId ); // phpcs:ignore
			},
		] );
	} );
}

/**
 * Add group features
 *
 * @param string $key         Feature key.
 * @param string $meta_key    Meta key.
 * @param string $title       Feature title.
 * @param string $description Feature description.
 */
function add_group_feature( string $key, string $meta_key, string $title, string $description ): void {
	/**
	 * Add group features
	 */
	\add_filter( 'nisje_group_features', function( array $features ) use ( $key, $title, $description ): array {
		$features[ $key ] = [
			'label'       => $title,
			'description' => $description,
		];

		return $features;
	} );

	/**
	 * Extend BuddyPress Groups REST API.
	 */
	\add_action( 'rest_api_init', function() use ( $key, $meta_key, $description ): void {
		\bp_rest_register_field( 'groups', $key, [
			'get_callback' => function( $object ) use ( $meta_key ) {
				return (bool) \groups_get_groupmeta( $object['id'], $meta_key, true );
			},
			'schema'       => [
				'context'     => [ 'view', 'edit' ],
				'description' => $description,
				'type'        => 'bool',
				'readonly'    => true,
			],
		] );
	} );

	/**
	 * Set group meta
	 */
	\add_action( 'nisje_set_group_meta', function( $group, $request, $group_args ) use ( $key, $meta_key ): void {
		if ( isset( $group_args[ $key ] ) && $group_args[ $key ] ) {
			\groups_update_groupmeta( $group->id, $meta_key, true );
		}
	}, 10, 3 );
}
