<?php
/**
 * Nisje Digest Rest API.
 *
 * @package Nisje
 */

declare( strict_types = 1 );
namespace Nisje\Extension\Digest\Rest;

use function Nisje\Utils\string_ends_with;

/**
 * Create Nisje Digest rest api endpoint.
 */
class Digest extends \WP_REST_Controller {
	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes(): void {
		$namespace = 'nisje/v1';
		$base      = 'digest';

		\register_rest_route( $namespace, '/' . $base, [
			[
				'methods'             => \WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_item' ],
				'permission_callback' => [ $this, 'get_item_permissions_check' ],
				'args'                => [
					'context' => [
						'default' => 'view',
					],
				],
			],
		] );
	}

	/**
	 * Get one item from the collection
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_Error|\WP_REST_Response
	 */
	public function get_item( $request ): \WP_REST_Response { // phpcs:ignore
		$start_date = $request['start'] ?: '-2 days 23:59:59';
		$end_date   = $request['end'] ?: '-1 days 23:59:59';

		$groups   = [];
		$posts    = [];
		$response = [
			'home'  => \home_url( '/' ),
			'title' => \get_bloginfo( 'name' ),
			'date'  => \gmdate( 'j. F Y', strtotime( $end_date ) ),
			'posts' => [],
		];

		$posts_query = new \WP_Query( [
			'post_type'           => 'nisje-user-content',
			'post_status'         => 'publish',
			'posts_per_page'      => -1,
			'ignore_sticky_posts' => true,
			'date_query'          => [
				'after'  => $start_date,
				'before' => $end_date,
			],
		] );

		$excerpt_more = \apply_filters( 'excerpt_more', ' [&hellip;]' );

		if ( $posts_query->have_posts() ) {
			while ( $posts_query->have_posts() ) {
				$posts_query->the_post();
				$id     = \get_the_ID();
				$tags   = [];
				$images = [];

				$terms = \get_the_terms( $id, 'nisje-tags' );
				if ( $terms ) {
					foreach ( $terms as $term ) {
						$tags[] = [
							'id'   => $term->term_id,
							'name' => $term->name,
						];
					}
				}

				$post_images = \get_post_meta( $id, 'nisje-images', true );
				if ( ! empty( $post_images ) ) {
					$images_count = count( $post_images );

					for ( $i = 0; $i < $images_count; $i++ ) {
						if ( $i < 4 ) {
							$images[] = \wp_get_attachment_image_src( $post_images[ $i ], 'medium' )[0];
						}
					}
				}

				$content = \wp_trim_excerpt();

				if ( string_ends_with( $content, $excerpt_more ) ) {
					$content .= sprintf( ' <a href="%s" style="text-decoration: underline">Continue reading</a>', home_url( '/activity/' . $id ) );
				}

				$posts[] = [
					'id'             => $id,
					'group_id'       => \wp_get_post_parent_id( $id ),
					'content'        => $content,
					'author'         => \get_the_author(),
					'author_url'     => \home_url( '/members/' . \get_the_author_meta( 'ID' ) ),
					'comments_count' => \get_comments_number(),
					'tags'           => $tags,
					'images'         => $images,
				];
			}
		}

		$group_ids = array_values( array_unique( \wp_list_pluck( $posts, 'group_id' ) ) );

		// Remove 0 from group ids.
		$group_zero = array_search( 0, $group_ids, true );
		if ( false !== $group_zero ) {
			unset( $group_ids[ $group_zero ] );
		}

		$digest_groups = \groups_get_groups( [
			'include'     => $group_ids,
			'show_hidden' => true,
		] );

		$groups_colors = [
			'#29a873',
			'#dc2221',
			'#fab901',
			'#fae100',
			'#d40f7d',
			'#6e1d78',
			'#006db5',
			'#1abae9',
			'#97be0c',
			'#333',
		];

		foreach ( $digest_groups['groups'] as $group ) {
			$color_index          = (int) substr( (string) $group->id, -1 );
			$groups[ $group->id ] = [
				'name'   => $group->name,
				'avatar' => \bp_core_fetch_avatar( [
					'item_id' => $group->id,
					'object'  => 'group',
					'type'    => 'full',
					'html'    => false,
				] ),
				'color'  => $groups_colors[ $color_index ],
			];
		}

		foreach ( $posts as $post ) {
			$post['group']       = isset( $groups[ $post['group_id'] ] ) ? $groups[ $post['group_id'] ] : '';
			$response['posts'][] = $post;
		}

		$data = $this->prepare_item_for_response( $response, $request );

		return new \WP_REST_Response( $data, 200 );
	}

	/**
	 * Check if a given request has access to get a specific item
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_Error|bool
	 */
	public function get_item_permissions_check( $request ) { // phpcs:ignore
		return \is_user_logged_in() && \current_user_can( 'manage_options' );
	}

	/**
	 * Prepare the item for the REST response
	 *
	 * @param mixed            $items   WordPress representation of the item.
	 * @param \WP_REST_Request $request Request object.
	 * @return mixed
	 */
	public function prepare_item_for_response( $items, $request ) { // phpcs:ignore
		return $items;
	}
}
