<?php
/**
 * Setup Open Community Component
 *
 * @package Nisje
 */

declare( strict_types = 1 );
namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Open Community Component Class
 */
class Open_Community extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		$path  = trailingslashit( dirname( __FILE__ ) );
		$label = esc_html__( 'Open Community', 'nisje-open-community' );

		parent::start(
			'open-community',
			$label,
			$path
		);
	}

	/**
	 * Setup globals
	 */
	public function setup_globals() {
		$settings = get_option( $this->option_key );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			foreach ( $settings as $key => $value ) {
				$this->settings[ $key ] = $value;
			}
		}

		$defaults = [
			'name'              => $this->name,
			'option_key'        => $this->option_key,
			'field_key_prefix'  => 'nisje_open_community',
			'settings_menu_key' => 'nisje-settings-open-community',
		];

		$this->settings = array_merge( $defaults, (array) $this->settings );
	}

	/**
	 * Includes
	 */
	public function includes() {
		$this->include_file( 'helper-functions.php' );
	}

	/**
	 * Setup Filters
	 */
	public function setup_filters() {
		add_filter( 'nisje_filter_routes', [ $this, 'filter_open_routes' ] );
		add_filter( 'nisje_redirect_to_login', '__return_false' );

		if ( is_admin() ) {
			add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_group' ] );
			add_action( 'nisje_settings_add_option_page', [ $this, 'setup_settings_menu' ] );
			add_action( 'nisje_settings_populate_options_from_acf', [ $this, 'populate_acf_settings' ] );
		}
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		$this->include_file( 'rest-handlers/class-open-community-controller.php' );
		$controller = new \Dekode\Nisje\Components\Rest\Open_Community_Controller();
		$controller->register_routes();
	}

	/**
	 * Filter open routes
	 *
	 * @param array $open_routes Open routes.
	 * @return array $open_routes Open routes.
	 */
	public function filter_open_routes( $open_routes ) {
		if ( ! empty( $open_routes ) || ! is_array( $open_routes ) ) {
			array_push( $open_routes, '/nisje/v1/open-community' );
		}

		return $open_routes;
	}

	/**
	 * Populate settings
	 */
	public function register_public_settings() {
		$this->public_settings = [
			'guest_user_id' => nisje_open_community_get_guest_user_id(),
		];
	}

	/**
	 * Setup the admin page.
	 */
	public function setup_settings_menu() {
		$label = esc_html__( 'Open Community', 'nisje-open-community' );

		acf_add_options_sub_page( [
			'page_title'  => $label,
			'menu_title'  => $label,
			'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
			'menu_slug'   => nisje_get_setting( 'settings_menu_key', $this->settings_key ),
			'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
		] );
	}

	/**
	 * Include acf field groups.
	 */
	public function register_field_group() {
		$this->include_file( 'acf/field-groups/acf-field-group-options-open-community.php' );
	}

	/**
	 * Populate ACF Options General
	 *
	 * @param \WP_Screen $screen Screen object.
	 */
	public function populate_acf_settings( \WP_Screen $screen ) {
		if ( true !== strpos( $screen->id, nisje_get_setting( 'settings_menu_key', $this->settings_key ) ) ) {
			$field_key_prefix = nisje_get_setting( 'field_key_prefix', $this->settings_key );
			$guest_user_field = get_field( $field_key_prefix . '_guest_user', 'option' );

			$settings = [
				'guest_user' => $guest_user_field,
			];

			update_option( $this->option_key, $settings );
		}
	}
}
