<?php
/**
 * Open Community endpoints.
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components\Rest;

use \Firebase\JWT\JWT;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * User Switching Rest Class.
 */
class Open_Community_Controller extends \WP_REST_Controller {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace = 'nisje/v1';
		$this->rest_base = 'open-community';
		$this->hook_base = 'open-community';
	}

	/**
	 * Register the routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/', [
				[
					'methods'             => \WP_REST_Server::CREATABLE,
					'permission_callback' => [ $this, 'generate_token_permissions_check' ],
					'callback'            => [ $this, 'generate_token' ],
					'args'                => [
						'user_id' => [
							'description' => esc_html__( 'Unique identifier for the object.', 'nisje' ),
							'type'        => 'integer',
						],
					],
				],
				'schema' => [ $this, 'get_public_item_schema' ],
			]
		);
	}

	/**
	 * Check if a given request has access to auto login.
	 *
	 * @return WP_Error|boolean
	 */
	public function generate_token_permissions_check() {
		if ( ! class_exists( '\\Firebase\\JWT\\JWT' ) ) {
			return new \WP_Error(
				'nisje_bad_config', esc_html__( 'JWT is not configurated properly, please contact the admin', 'nisje' ), [
					'status' => 403,
				]
			);
		}

		$secret_key = defined( 'JWT_AUTH_SECRET_KEY' ) ? JWT_AUTH_SECRET_KEY : false;
		if ( ! $secret_key ) {
			return new \WP_Error(
				'nisje_auth_bad_config', esc_html__( 'JWT is not configurated properly, please contact the admin', 'nisje' ), [
					'status' => 403,
				]
			);
		}

		return true;
	}

	/**
	 * Generate token.
	 *
	 * @param WP_REST_Request $request Rest Request.
	 * @return WP_REST_Request|WP_Error Plugin object data on success, WP_Error otherwise.
	 */
	public function generate_token( $request ) {
		$user = get_userdata( nisje_open_community_get_guest_user_id() );

		if ( ! $user instanceof \WP_User ) {
			return new \WP_Error(
				'nisje_no_user', esc_html__( 'No user object.', 'nisje' ), [
					'status' => 403,
				]
			);
		}

		/** Valid credentials, the user exists create the according Token */
		$issued     = time();
		$not_before = $issued;
		// Only valid 1 day.
		$expire = $issued + DAY_IN_SECONDS;

		$token = [
			'iss'  => get_bloginfo( 'url' ),
			'iat'  => $issued,
			'nbf'  => $not_before,
			'exp'  => $expire,
			'data' => [
				'user' => [
					'id' => $user->data->ID,
				],
			],
		];

		$secret_key = defined( 'JWT_AUTH_SECRET_KEY' ) ? JWT_AUTH_SECRET_KEY : false;
		$token      = \Firebase\JWT\JWT::encode( $token, $secret_key );
		$data       = [
			'token'             => $token,
			'user_email'        => $user->data->user_email,
			'user_nicename'     => $user->data->user_nicename,
			'user_display_name' => $user->data->display_name,
		];

		$response = rest_ensure_response( $data );

		return $response;
	}
}
