<?php
/**
 * Merge content RS.
 *
 * @package Nisje
 */

declare( strict_types=1 );

namespace Dekode\Nisje\Commands;

use WP_CLI;
use Exception;
use StdClass;
use WP_Post;

if ( class_exists( 'WP_CLI' ) ) {
	/**
	 * Merge old intranett content before Nisje era
	 */
	class RS_Merge_Content {
		/**
		 * This function is run on the command "wp rs-merge-content merge_activities"
		 */
		public function merge_activities() {
			add_filter( 'wp_revisions_to_keep', [ $this, 'restrict_revisions' ], 10, 2 );

			$query_args = [
				'display_comments' => 'threaded',
				'sort'             => 'ASC',
				'filter'           => [
					'action' => [
						'activity_update',
					],
				],
				'per_page'         => '20000',
				'show_hidden'      => true,
			];

			$activities = bp_activity_get( $query_args );

			if ( empty( $activities['activities'] ) ) {
				WP_CLI::error( 'Could not find any activities' );
			}

			WP_CLI::line( 'Found ' . count( $activities['activities'] ) . ' activities.' );

			foreach ( $activities['activities'] as $item ) {
				WP_CLI::line( 'Importing activity ' . $item->type . ' (' . $item->id . ')' );
				$this->create_or_update( $item );
			}

			WP_CLI::success( 'Activities successfully merged and updated' );
		}

		/**
		 * This function is run on the command "wp rs-merge-content merge_reactions"
		 */
		public function merge_reactions() {
			$query_args = [
				'sort'        => 'ASC',
				'filter'      => [
					'action' => [
						'bp_activity_reaction_like',
					],
				],
				'per_page'    => '20000',
				'show_hidden' => true,
			];

			$reactions = bp_activity_get( $query_args );

			if ( empty( $reactions['activities'] ) ) {
				WP_CLI::error( 'Could not find any reactions' );
			}

			WP_CLI::line( 'Found ' . count( $reactions['activities'] ) . ' reactions.' );

			foreach ( $reactions['activities'] as $item ) {
				WP_CLI::line( 'Importing reaction ' . $item->type . ' (' . $item->id . ')' );
				$this->add_reaction( $item );
			}

			WP_CLI::success( 'Reactions successfully merged and updated' );
		}

		/**
		 * This function is run on the command "wp rs-merge-content merge_rikkes"
		 */
		public function merge_rikke() {
			$args = [
				'post_type'     => 'di-mp',
				'post_per_page' => -1,
			];

			$group_id = groups_create_group( [
				'group_id'     => 0,
				'creator_id'   => 2,
				'name'         => 'Rikkes Hjørne',
				'description'  => 'Følg generalsekretærens blogg!',
				'status'       => 'public',
				'parent_id'    => 0,
				'enable_forum' => 0,
			] );

			groups_update_groupmeta( $group_id, '_nisje_blogs_enabled', true );

			WP_CLI::line( 'Rikkes group created ' . $group_id );

			$query = new \WP_Query( $args );
			if ( $query->have_posts() ) {
				while ( $query->have_posts() ) {
					$query->the_post();

					$post = get_post();

					$post->post_type   = 'nisje-blogs-group';
					$post->post_parent = $group_id;
					$post->post_author = 1049;

					wp_update_post( $post );
					update_post_meta( $post->ID, '_nisje_group_connection', $group_id );
					WP_CLI::line( 'Merged post ' . $post->ID );
				}

				WP_CLI::success( 'Rikke is merged!' );
			} else {
				WP_CLI::error( 'Could not find any Rikke Posts' );
			}
			$posts = $query->get_posts();
			if ( isset( $posts[0] ) ) {
				return $posts[0];
			} else {
				return null;
			}
		}

		/**
		 * Merge options
		 */
		public function merge_options() {
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.serialize_unserialize
			$options = unserialize( 'a:65:{i:0;b:0;s:18:"custom_css_post_id";i:-1;s:18:"nav_menu_locations";a:1:{s:7:"primary";i:18;}s:14:"intranet_style";s:6:"square";s:24:"intranet_masterbar_width";s:7:"content";s:22:"intranet_content_width";s:7:"content";s:13:"intranet_font";s:4:"lato";s:24:"mastbar_background_color";s:7:"#f1f1f1";s:27:"navigation_background_color";s:7:"#ffffff";s:21:"navigation_text_color";s:7:"#444444";s:21:"navigation_icon_color";s:7:"#adadad";s:27:"navigation_icon_hover_color";s:7:"#000000";s:27:"navigation_text_hover_color";s:7:"#000000";s:25:"navigation_selected_color";s:7:"#ff0000";s:31:"mastbar_background_color_mobile";s:7:"#ffffff";s:25:"mastbar_text_color_mobile";s:7:"#333333";s:27:"mastbar_border_color_mobile";s:7:"#dddddd";s:26:"bookmarks_background_color";s:7:"#dd3333";s:20:"bookmarks_text_color";s:7:"#ffffff";s:23:"events_background_color";s:7:"#ffffff";s:28:"events_date_background_color";s:7:"#1e73be";s:22:"events_date_text_color";s:7:"#ffffff";s:31:"news_frontpage_background_color";s:7:"#061534";s:29:"news_archive_background_color";s:7:"#061534";s:23:"news_archive_text_color";s:7:"#ffffff";s:11:"custom_logo";i:9961;s:29:"mp_frontpage_background_color";s:7:"#fcfcfc";s:23:"mp_frontpage_text_color";s:7:"#444444";s:27:"mp_archive_background_color";s:7:"#fcfcfc";s:21:"mp_archive_text_color";s:7:"#454545";s:17:"fp_alert_bg_color";s:7:"#dd3333";s:13:"nav-bg-narrow";s:7:"#030d35";s:11:"nav-bg-main";s:7:"#030d35";s:18:"nav-group-share-bg";s:7:"#ffffff";s:20:"nav-group-share-text";s:7:"#000000";s:13:"nav-group-nav";s:7:"#f1f1f1";s:14:"nav-share-text";s:7:"#36373b";s:13:"nav-bullet-bg";s:7:"#db3539";s:15:"nav-selected-bg";s:7:"#363c52";s:15:"nav-left-border";s:7:"#db3539";s:14:"nav-icon-hover";s:7:"#b5b5b5";s:12:"nav-hover-bg";s:7:"#db3539";s:25:"box-top-leve-editorial-bg";s:7:"#030d35";s:22:"box-top-leve-editorial";s:7:"#ffffff";s:17:"circle-event-date";s:7:"#2038a2";s:10:"main-icons";s:7:"#010082";s:14:"nav-icons-main";s:7:"#ffffff";s:9:"button-bg";s:7:"#2038a2";s:13:"button-border";s:7:"#2038a2";s:7:"tags-bg";s:7:"#2038a2";s:9:"edit-icon";s:7:"#2038a2";s:10:"nav-border";s:7:"#030d35";s:15:"button-bg-hover";s:7:"#4b60c3";s:19:"button-border-hover";s:7:"#4b60c3";s:13:"tags-bg-hover";s:7:"#4b60c3";s:19:"shortcut-icon-color";s:7:"#db3539";s:12:"shortcut-box";s:7:"#db3539";s:13:"shortcut-text";s:7:"#ffffff";s:19:"shortcut-text-hover";s:7:"#dbdbdb";s:16:"shortcut-icon-bg";s:7:"#ffffff";s:10:"arrow-icon";s:7:"#ffffff";s:18:"nav-share-bg-hover";s:7:"#ffffff";s:20:"nav-share-text-hover";s:7:"#36373b";s:23:"circle-event-date-color";s:7:"#ffffff";s:10:"nisje_font";s:8:"circular";}' );
			unset( $options[0] );

			unset( $options['custom_css_post_id'] );

			update_option( 'theme_mods_intranet', $options );
		}

		/**
		 * Restrict revisions when importing.
		 *
		 * @param int      $num  Number of allowed revisions.
		 * @param \WP_Post $post Current post.
		 *
		 * @return int Custom revisions.
		 */
		public function restrict_revisions( $num, $post ) {
			return 0;
		}

		/**
		 * Takes an activity and creates a new post for it or updates an existing one.
		 *
		 * @param StdClass $item The activity.
		 */
		private function create_or_update( StdClass $item ) {
			$original_post = $this->get_existing_post( $item->id );

			$args = [
				'post_author'  => isset( $item->user_id ) && $item->user_id ? (int) $item->user_id : 1,
				'post_title'   => '',
				'post_content' => isset( $item->content ) ? $item->content : '',
				'post_status'  => 'publish',
				'post_type'    => nisje_get_setting( 'post_type_name', 'user-content' ),
			];

			if ( isset( $item->item_id ) && $item->item_id ) {
				$args['post_parent'] = $item->item_id;
			}

			if ( isset( $item->date_recorded ) ) {
				$args['post_date'] = $item->date_recorded;
			}

			if ( is_object( $original_post ) && 'WP_Post' === get_class( $original_post ) ) {
				$args['ID'] = $original_post->ID;
				WP_CLI::line( "\t" . 'Found original post so updating post ' . $original_post->ID );
			} else {
				WP_CLI::line( "\t" . 'No original post found, creating new' );
			}

			$post_id = wp_insert_post( $args );

			if ( isset( $item->item_id ) && $item->item_id ) {
				update_post_meta( $post_id, '_nisje_group_connection', $item->item_id );
			}

			$taxonomy_format_name = nisje_get_setting( 'taxonomy_format_name', 'user-content' );
			$taxonomy_type_name   = nisje_get_setting( 'taxonomy_type_name', 'user-content' );

			$images = bp_activity_get_meta( $item->id, 'di-images', true );
			if ( ! empty( $images ) && is_array( $images ) ) {
				$checked_images = [];

				foreach ( $images as $image ) {
					if ( isset( $image ) && $image ) {
						$checked_images[] = $image;
					}
				}

				update_post_meta( $post_id, 'nisje-images', $checked_images );
				wp_set_post_terms( $post_id, [ 'image', 'gallery' ], $taxonomy_format_name );
			} else {
				$featured_image = bp_activity_get_meta( $item->id, 'post_image', true );
				if ( isset( $featured_image['id'] ) ) {
					$images = bp_activity_get_meta( $item->id, 'di-images', true );
					if ( ! empty( $images ) && is_array( $images ) ) {
						$images[] = $featured_image['id'];
					} else {
						update_post_meta( $post_id, 'nisje-images', [ $featured_image['id'] ] );
						wp_set_post_terms( $post_id, [ 'image' ], $taxonomy_format_name );
					}

					set_post_thumbnail( $post_id, $featured_image['id'] );
				}
			}

			$video = bp_activity_get_meta( $item->id, 'di-video', true );
			if ( $video && ! empty( $video ) ) {
				update_post_meta( $post_id, 'nisje-video', $video );
				wp_set_post_terms( $post_id, 'video', $taxonomy_format_name );
			}

			$share = bp_activity_get_meta( $item->id, 'di-share', true );
			if ( $share && is_array( $share ) ) {
				update_post_meta( $post_id, 'nisje-share', $share );
				wp_set_post_terms( $post_id, 'link', $taxonomy_format_name );
			}

			if ( isset( $item->children ) && is_array( $item->children ) && ! empty( $item->children ) ) {
				$this->add_comments( $item->children, $post_id );
			}

			update_post_meta( $post_id, 'nisje_old_activity_id', $item->id );

			wp_set_post_terms( $post_id, 'social', $taxonomy_type_name );
		}

		/**
		 * Finds existing post in WordPress
		 *
		 * @param int $remote_id The ID of the post to be found.
		 *
		 * @return array|boolean
		 */
		private function get_existing_post( int $remote_id ) {
			$args = [
				'post_type'  => nisje_get_setting( 'post_type_name', 'user-content' ),
				'meta_query' => [ // phpcs:ignore
					[
						'key'     => 'nisje_old_activity_id',
						'value'   => $remote_id,
						'compare' => '=',
						'type'    => 'numeric',
					],
				],
			];

			$query = new \WP_Query( $args );

			$posts = $query->get_posts();
			if ( isset( $posts[0] ) ) {
				return $posts[0];
			} else {
				return null;
			}
		}

		/**
		 * Finds existing post in WordPress
		 *
		 * @param int $remote_id The ID of the post to be found.
		 *
		 * @return array|boolean
		 */
		private function get_existing_comment( int $remote_id ) {
			$args = [
				'meta_query' => [ // phpcs:ignore
					[
						'key'     => 'nisje_old_comment_id',
						'value'   => $remote_id,
						'compare' => '=',
						'type'    => 'numeric',
					],
				],
			];

			$comments = \get_comments( $args );

			if ( isset( $comments[0] ) ) {
				return $comments[0];
			} else {
				return null;
			}
		}

		/**
		 * Add comments
		 *
		 * @param array $comments Comments.
		 * @param int   $post_id  Post ID.
		 */
		private function add_comments( array $comments, int $post_id ) {
			foreach ( $comments as $comment ) {
				$original_comment = $this->get_existing_comment( $comment->id );

				$data = [
					'comment_post_ID'      => $post_id,
					'comment_author'       => $comment->display_name,
					'comment_author_email' => $comment->user_email,
					'comment_content'      => $comment->content,
					'comment_parent'       => 0,
					'user_id'              => $comment->user_id,
					'comment_date'         => $comment->date_recorded,
					'comment_approved'     => 1,
				];

				if ( is_object( $original_comment ) && 'WP_Comment' === get_class( $original_comment ) ) {
					$data['comment_ID'] = $original_comment->comment_ID;
					WP_CLI::line( "\t" . 'Found original comment so updating comment ' . $original_comment->comment_ID . ' for post ' . $post_id );
					wp_update_comment( $data );
					$comment_id = $original_comment->comment_ID;
				} else {
					WP_CLI::line( "\t" . 'No original comment found, creating new' );
					$comment_id = wp_insert_comment( $data );
					update_comment_meta( $comment_id, 'nisje_old_comment_id', $comment->id );
				}
			}
		}

		/**
		 * Add reactions
		 *
		 * @param StdClass $reaction Reactions.
		 */
		private function add_reaction( StdClass $reaction ) {
			$original_post = $this->get_existing_post( $reaction->item_id );
			if ( is_object( $original_post ) && 'WP_Post' === get_class( $original_post ) ) {
				nisje_add_reaction( $original_post->ID, $reaction->user_id, 'post', 1 );
				WP_CLI::line( "\t" . 'Found post so adding reaction ' . $original_post->ID . ' reaction id ' . $reaction->id . ' item id ' . $reaction->item_id );

				return;
			}

			$post = bp_activity_get_meta( $reaction->item_id, 'post_id', true );
			if ( $post ) {
				$tmp_post = get_post( $post );
				if ( $tmp_post instanceof \WP_Post ) {
					nisje_add_reaction( $tmp_post->ID, $reaction->user_id, 'post', 1 );
					WP_CLI::line( "\t" . 'Found post so adding reaction ' . $tmp_post->ID . ' reaction id ' . $reaction->id . ' item id ' . $reaction->item_id );

					return;
				}
			}

			WP_CLI::line( "\t" . 'Did not find any connected posts ' . $reaction->id . ' item id ' . $reaction->item_id );
		}

		/**
		 * Adds the hostname and scheme to relative URLs.
		 *
		 * @param string $text The string to perform the replacement in.
		 *
		 * @return string
		 */
		private function make_urls_absolute( string $text ): string {
			$text = preg_replace( '/<img(.*)src="\/(.*)"(.*)>/u', '<img\\1src="' . $this->api_url . '/\\2"\\3"', $text );

			return $text;
		}

	}

	$merge = new RS_Merge_Content();

	WP_CLI::add_command( 'rs-merge-content', $merge );
}
