<?php
/**
 * Retriever Import
 *
 * @package Nisje
 */

declare( strict_types=1 );

namespace Dekode\Nisje\Commands;

use WP_CLI;
use Exception;
use StdClass;
use WP_Post;

if ( class_exists( 'WP_CLI' ) ) {
	/**
	 * Retriever import.
	 */
	class RS_Retriever {
		/**
		 * This function is run on the command "wp rs-retriever import"
		 */
		public function import() {
			$url = 'https://www.retriever-info.com/feed/2002482/intranett/index.xml';

			add_filter( 'wp_revisions_to_keep', [ $this, 'restrict_revisions' ], 10, 2 );

			$load_xml = simplexml_load_file( $url );

			if ( ! $load_xml ) {
				WP_CLI::error( 'Could not find any posts' );
			}

			WP_CLI::line( 'Found ' . count( $load_xml->channel->item ) . ' posts.' );

			foreach ( $load_xml->channel->item as $item ) {
				$guid = (string) $item->guid;
				WP_CLI::line( 'Importing post ' . $guid );
				$this->create_or_update( $item );
			}

			WP_CLI::success( 'Retriever posts successfully merged and updated' );
		}

		/**
		 * Restrict revisions when importing.
		 *
		 * @param int      $num  Number of allowed revisions.
		 * @param \WP_Post $post Current post.
		 *
		 * @return int Custom revisions.
		 */
		public function restrict_revisions( $num, $post ) {
			return 0;
		}

		/**
		 * Takes a retriever post and creates a new post for it or updates an existing one.
		 *
		 * @param \SimpleXMLElement $item The item.
		 */
		private function create_or_update( \SimpleXMLElement $item ) {
			$guid = (string) $item->guid;

			$original_post = $this->get_existing_post( $guid );
			$author        = $this->get_import_author();

			$title       = (string) $item->title;
			$link        = (string) $item->link;
			$description = (string) $item->description;
			$pubdate     = (string) $item->pubDate; // phpcs:ignore

			$dc      = $item->children( 'http://purl.org/dc/elements/1.1/' );
			$ret     = $item->children( 'http://www.retriever-info.com' );
			$creator = (string) $dc->creator;

			$ret_namespace = [
				'docdate'     => isset( $ret->docdate ) ? (string) $ret->docdate : '',
				'source'      => isset( $ret->source ) ? (string) $ret->source : '',
				'source_id'   => isset( $ret->source_id ) ? (string) $ret->source_id : '',
				'mediatype'   => isset( $ret->mediatype ) ? (string) $ret->mediatype : '',
				'language'    => isset( $ret->language ) ? (string) $ret->language : '',
				'wordcount'   => isset( $ret->wordcount ) ? (string) $ret->wordcount : '',
				'byline'      => isset( $ret->byline ) ? (string) $ret->byline : '',
				'subheadline' => isset( $ret->subheadline ) ? (string) $ret->subheadline : '',
				'imgtext'     => isset( $ret->imgtext ) ? (string) $ret->imgtext : '',
				'pdfurl'      => isset( $ret->pdfurl ) ? (string) $ret->pdfurl : '',
			];

			$dynteaser = (string) $ret->dynteaser;
			$date_gmt  = gmdate( 'Y-m-d H:i:s', strtotime( $pubdate ) );
			$date      = mysql2date( 'Y-m-d H:i:s', $date_gmt );

			$data = [
				'post_status'   => 'publish',
				'post_type'     => nisje_get_setting( 'post_type_name', 'rs_retriever' ),
				'ping_status'   => 'closed',
				'post_author'   => $author,
				'post_title'    => $title,
				'post_content'  => $description,
				'post_excerpt'  => $dynteaser,
				'post_date_gmt' => $date_gmt,
				'post_date'     => $date,
			];

			if ( is_object( $original_post ) && 'WP_Post' === get_class( $original_post ) ) {
				$data['ID'] = $original_post->ID;
				WP_CLI::line( "\t" . 'Found original post so updating post ' . $original_post->ID );
			} else {
				WP_CLI::line( "\t" . 'No original post found, creating new' );
			}

			$post_id = wp_insert_post( $data );

			if ( ! $post_id ) {
				WP_CLI::line( "\t" . 'Post not imported - ' . $title );
			}

			$news_category = $this->get_import_category();

			wp_set_object_terms( $post_id, $creator, nisje_get_setting( 'taxonomy_pub_name', 'rs_retriever' ), true );
			wp_set_object_terms( $post_id, $news_category, nisje_get_setting( 'taxonomy_category_name', 'news' ), true );

			update_post_meta( $post_id, 'retriever_link', esc_url( $link ) );
			update_post_meta( $post_id, 'nisje_rs_retriever_guid', $guid );

			$retriver_object = [];
			foreach ( $ret_namespace as $key => $ret_value ) {
				$retriever_object[ $key ] = $ret_value;
			}

			update_post_meta( $post_id, 'retriever_object', $retriever_object );
		}

		/**
		 * Finds existing post in WordPress
		 *
		 * @param string $remote_id The ID of the post to be found.
		 *
		 * @return array|boolean
		 */
		private function get_existing_post( string $remote_id ) {
			$args = [
				'post_type'  => nisje_get_setting( 'post_type_name', 'rs_retriever' ),
				'meta_query' => [ // phpcs:ignore
					[
						'key'     => 'nisje_rs_retriever_guid',
						'value'   => $remote_id,
						'compare' => '=',
						'type'    => 'string',
					],
				],
			];

			$query = new \WP_Query( $args );

			$posts = $query->get_posts();
			if ( isset( $posts[0] ) ) {
				return $posts[0];
			} else {
				return null;
			}
		}

		/**
		 * Import author id
		 *
		 * @return int Author id.
		 */
		private function get_import_author() : int {
			return nisje_get_setting( 'import_author', 'rs_retriever' );
		}

		/**
		 * Import news category
		 *
		 * @return int Catgory id.
		 */
		private function get_import_category() : int {
			return nisje_get_setting( 'news_category', 'rs_retriever' );
		}
	}

	$retriever = new RS_Retriever();

	WP_CLI::add_command( 'rs-retriever', $retriever );
}
