<?php
/**
 * Setup Segment Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

if ( ! class_exists( '\\Dekode\\Nisje\\Components\\Segment' ) ) {

	/**
	 * Segment Component Class
	 */
	class Segment extends Component {
		/**
		 * Constructor
		 */
		public function __construct() {
			$path = trailingslashit( dirname( __FILE__ ) );

			parent::start(
				'segment',
				esc_html__( 'Segment', 'nisje-segment' ),
				$path
			);
		}

		/**
		 * Setup globals
		 */
		public function setup_globals() {
			$settings = get_option( $this->option_key );
			if ( is_array( $settings ) && ! empty( $settings ) ) {
				foreach ( $settings as $key => $value ) {
					$this->settings[ $key ] = $value;
				}
			}

			$defaults = [
				'name'              => $this->name,
				'option_key'        => $this->option_key,
				'field_key_prefix'  => 'nisje_segment',
				'settings_menu_key' => 'nisje-settings-segment',
			];

			$this->settings = array_merge( $defaults, (array) $this->settings );
		}

		/**
		 * Setup Filters
		 */
		public function setup_filters() {
			if ( is_admin() ) {
				add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_group' ] );
				add_action( 'nisje_settings_add_option_page', [ $this, 'setup_settings_menu' ] );
				add_action( 'nisje_settings_populate_options_from_acf', [ $this, 'populate_acf_settings' ] );
			}

			add_filter( 'nisje_frontend_options', [ $this, 'add_frontend_option' ] );
		}

		/**
		 * Add tracking code to options
		 *
		 * @param array $options Options.
		 */
		public function add_frontend_option( $options ) {
			$code = nisje_get_setting( 'tracking_code', 'segment' );

			if ( ! empty( $code ) ) {
				$options['segmentTrackingCode'] = $code;
			}

			return $options;
		}

		/**
		 * Setup the admin page.
		 */
		public function setup_settings_menu() {
			acf_add_options_sub_page(
				[
					'page_title'  => $this->label,
					'menu_title'  => $this->label,
					'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
					'menu_slug'   => nisje_get_setting( 'settings_menu_key', $this->settings_key ),
					'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
				]
			);
		}

		/**
		 * Include acf field groups.
		 */
		public function register_field_group() {
			$this->include_file( 'acf/field-groups/acf-field-group-options-segment.php' );
		}

		/**
		 * Populate ACF Options General
		 *
		 * @param WP_Screen $screen Screen object.
		 */
		public function populate_acf_settings( $screen ) {
			if ( true !== strpos( $screen->id, nisje_get_setting( 'settings_menu_key', $this->settings_key ) ) ) {
				$segment = [
					'tracking_code' => '',
				];

				$field_key_prefix = nisje_get_setting( 'field_key_prefix', $this->settings_key );

				$segment['tracking_code'] = get_field( $field_key_prefix . '_tracking_code', 'option' );

				update_option( $this->option_key, $segment );
			}
		}
	}
}
