<?php
/**
 * Extend user content endpoint.
 *
 * @package Nisje
 */

$taxonomy_format_name = nisje_get_setting( 'taxonomy_format_name', 'user-content' );
$taxonomy_type_name   = nisje_get_setting( 'taxonomy_type_name', 'user-content' );

// Global Nisje core tags.
$taxonomy_tags_name = 'nisje-tags';

$extend_object = [
	nisje_get_setting( 'post_type_name', 'user-content' ),
	'feed',
];

// Title.
register_rest_field(
	$extend_object, 'title', [
		'get_callback' => function ( $object ) {
			$title = get_the_title( $object['id'] );
			if ( ! $title ) {
				return null;
			}

			$retval = [
				'raw'      => wp_strip_all_tags( $title ),
				'rendered' => $title,
			];

			return $retval;
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'The title of this user content post sans-HTML tags.', 'nisje-user-content' ),
			'type'        => 'object',
			'items'       => 'string',
			'readonly'    => true,
		],
	]
);

// Group ID.
register_rest_field(
	$extend_object, 'group_id', [
		'get_callback' => function ( $object ) {
			$ret_val = 0;

			if ( bp_is_active( 'groups' ) ) {
				$group_id = get_post_meta( $object['id'], '_nisje_group_connection', true );

				if ( $group_id ) {
					$ret_val = intval( $group_id );
				}
			}

			return $ret_val;
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'The ID of the group associated with this user content post.', 'nisje-user-content' ),
			'type'        => 'integer',
			'readonly'    => false,
		],
	]
);

// Group name.
register_rest_field(
	$extend_object, 'group_name', [
		'get_callback' => function ( $object ) {
			$ret_val = '';

			if ( bp_is_active( 'groups' ) ) {
				$group_id = get_post_meta( $object['id'], '_nisje_group_connection', true );
				$group    = nisje_get_group( $group_id );

				if ( ! is_wp_error( $group ) ) {
					$ret_val = bp_get_group_name( $group );
				}
			}

			return $ret_val;
		},
		'schema'       => [
			'context'     => [ 'view' ],
			'description' => esc_html__( 'The name of the group associated with this user content post.', 'nisje-user-content' ),
			'type'        => 'string',
			'readonly'    => true,
		],
	]
);

// Images.
register_rest_field(
	$extend_object, 'images', [
		'get_callback'    => function ( $object ) {
			$retval = [];

			$images = get_post_meta( $object['id'], 'nisje-images', true );
			if ( ! empty( $images ) && is_array( $images ) ) {
				foreach ( $images as $image ) {
					$retval[] = nisje_populate_image_data( $image );
				}
			}

			return $retval;
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_format_name ) {
			if ( $object instanceof \WP_Post ) {
				if ( ! empty( $field ) ) {
					update_post_meta( $object->ID, 'nisje-images', $field );
					wp_set_post_terms( $object->ID, [ 'image', 'gallery' ], $taxonomy_format_name );
				} else {
					delete_post_meta( $object->ID, 'nisje-images' );
					wp_remove_object_terms( $object->ID, [ 'image', 'gallery' ], $taxonomy_format_name );
				}
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Array of image src', 'nisje-user-content' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => false,
		],
	]
);

// Share.
register_rest_field(
	$extend_object, 'share', [
		'get_callback'    => function ( $object ) {
			$share = get_post_meta( $object['id'], 'nisje-share', true );
			if ( $share && is_array( $share ) ) {
				return $share;
			}

			return [];
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_format_name ) {
			if ( $object instanceof \WP_Post && ( ! empty( $field ) && is_array( $field ) ) ) {

				$share = [];

				if ( isset( $field['url'] ) ) {
					$share['url'] = esc_url( $field['url'] );
				}

				if ( isset( $field['title'] ) ) {
					$share['title'] = $field['title'];
				}

				if ( isset( $field['description'] ) ) {
					$share['description'] = $field['description'];
				}

				update_post_meta( $object->ID, 'nisje-share', $share );

				wp_set_post_terms( $object->ID, 'link', $taxonomy_format_name );
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Meta data from shared url', 'nisje-user-content' ),
			'type'        => 'object',
			'properties'  => [
				'url'   => [
					'context'     => [ 'view', 'edit' ],
					'description' => esc_html__( 'Url to link', 'nisje-user-content' ),
					'type'        => 'string',
					'format'      => 'uri',
				],
				'title' => [
					'context'     => [ 'view', 'edit' ],
					'description' => esc_html__( 'Title to link', 'nisje-user-content' ),
					'type'        => 'string',
				],
			],
		],
	]
);

// Video.
register_rest_field(
	$extend_object, 'video', [
		'get_callback'    => function ( $object ) {
			$video = get_post_meta( $object['id'], 'nisje-video', true );

			if ( $video ) {
				return esc_url( $video );
			}

			return '';
		},
		'update_callback' => function ( $field, $object ) use ( $taxonomy_format_name ) {
			if ( $object instanceof \WP_Post ) {
				update_post_meta( $object->ID, 'nisje-video', $field );

				wp_set_post_terms( $object->ID, 'video', $taxonomy_format_name );
			}
		},
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Video', 'nisje-user-content' ),
			'type'        => 'string',
			'format'      => 'url',
			'readonly'    => false,
		],
	]
);

register_rest_field(
	$extend_object, 'embed', [
		'get_callback' => function ( $object ) {
			$video = get_post_meta( $object['id'], 'nisje-video', true );

			if ( $video ) {
				return wp_oembed_get( $video );
			}

			return '';
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Embed', 'nisje-user-content' ),
			'type'        => 'string',
			'readonly'    => true,
		],
	]
);

// Formats.
register_rest_field(
	$extend_object, 'formats', [
		'get_callback' => function ( $object ) use ( $taxonomy_format_name ) {
			$formats = [];

			$terms = get_the_terms( $object['id'], $taxonomy_format_name );
			if ( $terms ) {
				foreach ( $terms as $term ) {
					$formats[ $term->term_id ] = $term->name;
				}
			}

			return $formats;
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Formats', 'nisje-user-content' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'string',
			],
			'readonly'    => true,
		],
	]
);

// Types.
register_rest_field( $extend_object, 'type', [
	'get_callback'    => function( $object ) use ( $taxonomy_type_name ) {
		$type = [];

		$terms = get_the_terms( $object['id'], $taxonomy_type_name );
		if ( $terms ) {
			$type = $terms[0]->slug;
		} else {
			$post = get_post( $object['id'] );
			if ( $post instanceof \WP_Post ) {
				$type = $post->post_type;
			}
		}

		return $type;
	},
	'update_callback' => function( $field, $object ) use ( $taxonomy_type_name ) {
		if ( $object instanceof \WP_Post ) {
			wp_set_post_terms( $object->ID, $field, $taxonomy_type_name );
		}
	},
	'schema'          => [
		'context'     => [ 'view', 'edit' ],
		'description' => esc_html__( 'The content type', 'nisje-user-content' ),
		'type'        => 'string',
		'enum'        => array_keys( nisje_get_setting( 'types', 'user-content' ) ),
		'required'    => true,
	],
] );

// Tags.
register_rest_field( $extend_object, 'tags', [
	'get_callback'    => function( $object ) use ( $taxonomy_tags_name ) {
		$tags = [];

		$terms = get_the_terms( $object['id'], $taxonomy_tags_name );
		if ( $terms ) {
			foreach ( $terms as $term ) {
				$tags[ $term->term_id ] = $term;
			}
		}

		return $tags;
	},
	'update_callback' => function( $field, $object ) use ( $taxonomy_tags_name ) {
		if ( $object instanceof \WP_Post ) {
			wp_set_post_terms( $object->ID, $field, $taxonomy_tags_name );
		}
	},
	'schema'          => [
		'context'     => [ 'view', 'edit' ],
		'description' => esc_html__( 'The content tags', 'nisje-user-content' ),
		'type'        => 'array',
		'items'       => [
			'type' => 'string',
		],
		'required'    => true,
	],
] );
