<?php
/**
 * User Content Endpoints
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components\Rest;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * User Controller Rest Class.
 */
class User_Content_Controller extends \WP_REST_Posts_Controller {
	/**
	 * Validate base permissions.
	 *
	 * @param WP_REST_Request $request  Full data about the request.
	 * @param string          $type     Create/Basic.
	 * @param int             $user_id  User ID to validate against.
	 * @param int             $group_id Group ID to validate against.
	 *
	 * @return bool|WP_Error True if user is logged in and defeats the checks else WP_Error
	 */
	private function base_permissions( $request, $type, $user_id = 0, $group_id = 0 ) {
		$auth = nisje_validate_rest_authentication( 'user-content', $type );
		if ( is_wp_error( $auth ) ) {
			return $auth;
		}

		if ( ! $user_id ) {
			$user_id = bp_loggedin_user_id();
		}

		if ( ! $group_id ) {
			if ( $request instanceof \WP_REST_Request && isset( $request['group_id'] ) ) {
				$group_id = (int) $request['group_id'];
			}
		}

		if ( in_array( $type, [ 'create_item', 'update_item' ], true ) ) {
			$excluded = nisje_validate_rest_excluded( $user_id );
			if ( is_wp_error( $excluded ) ) {
				return $excluded;
			}
		}

		if ( 'get_item' === $type ) {
			$post = $this->get_post( $request['id'] );
			if ( is_wp_error( $post ) ) {
				return $post;
			}

			if ( $post->post_parent && ! $group_id ) {
				return new \WP_Error(
					'nisje_rest_connection_error', esc_html__( 'Missing group ID.', 'nisje-user-content' ), [
						'status' => 404,
					]
				);
			}
		}

		if ( $group_id ) {
			$group = nisje_get_group( $group_id );
			if ( is_wp_error( $group ) ) {
				return $group;
			}

			if ( in_array( $type, [ 'get_item', 'update_item', 'delete_item' ], true ) ) {
				$group_id = (int) $request['group_id'];

				$post = $this->get_post( $request['id'] );
				if ( is_wp_error( $post ) ) {
					return $post;
				}

				if ( $post->post_parent !== $group_id ) {
					return new \WP_Error(
						'nisje_rest_not_connected_to_group', esc_html__( 'The post is not connected to this group.', 'nisje-user-content' ), [
							'status' => 403,
						]
					);
				}
			}

			if ( groups_is_user_banned( $user_id, $group->id ) ) {
				return new \WP_Error(
					'nisje_rest_banned_from_group', esc_html__( 'Sorry, you are banned from this group.', 'nisje-user-content' ), [
						'status' => 403,
					]
				);
			}

			if ( in_array( $group->status, [ 'hidden', 'private' ], true ) && ! groups_is_user_member( $user_id, $group->id ) ) {
				return new \WP_Error(
					'nisje_rest_not_member', esc_html__( 'Sorry, you need to be a member to see content from this group.', 'nisje-user-content' ), [
						'status' => rest_authorization_required_code(),
					]
				);
			}
		}

		return true;
	}

	/**
	 * Check if a given request has access to see posts.

	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_items_permissions_check( $request ) {
		$check_base = parent::get_items_permissions_check( $request );
		if ( is_wp_error( $check_base ) ) {
			return $check_base;
		}

		$extended_permission_check = $this->base_permissions( $request, 'get_items' );
		if ( is_wp_error( $extended_permission_check ) ) {
			return $extended_permission_check;
		}

		// Filter Query.
		add_filter( "rest_{$this->post_type}_query", [ $this, 'filter_rest_query_group_args' ], 10, 2 );

		return true;
	}

	/**
	 * Check if a given request has access to posts.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_item_permissions_check( $request ) {
		$check_base = parent::get_item_permissions_check( $request );
		if ( is_wp_error( $check_base ) ) {
			return $check_base;
		}

		$extended_permission_check = $this->base_permissions( $request, 'get_item' );
		if ( is_wp_error( $extended_permission_check ) ) {
			return $extended_permission_check;
		}

		return true;
	}

	/**
	 * Checks if a given request has access to create a post.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return true|WP_Error True if the request has access to create items, WP_Error object otherwise.
	 */
	public function create_item_permissions_check( $request ) {
		$check_base = parent::create_item_permissions_check( $request );
		if ( is_wp_error( $check_base ) ) {
			return $check_base;
		}

		$extended_permission_check = $this->base_permissions( $request, 'create_item' );
		if ( is_wp_error( $extended_permission_check ) ) {
			return $extended_permission_check;
		}

		// Set post<->group connections.
		add_filter( "rest_pre_insert_{$this->post_type}", [ $this, 'set_group_connection' ], 10, 2 );
		add_filter( "rest_insert_{$this->post_type}", [ $this, 'set_meta_group_connection' ], 10, 3 );

		return true;
	}

	/**
	 * Checks if a given request has access to update a post.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return true|WP_Error True if the request has access to update the item, WP_Error object otherwise.
	 */
	public function update_item_permissions_check( $request ) {
		$check_base = parent::update_item_permissions_check( $request );
		if ( is_wp_error( $check_base ) ) {
			return $check_base;
		}

		$group_permission_check = $this->base_permissions( $request, 'update_item' );
		if ( is_wp_error( $group_permission_check ) ) {
			return $group_permission_check;
		}

		// Set post<->group connections.
		add_filter( "rest_pre_insert_{$this->post_type}", [ $this, 'set_group_connection' ], 10, 2 );
		add_filter( "rest_insert_{$this->post_type}", [ $this, 'set_meta_group_connection' ], 10, 3 );

		return true;
	}

	/**
	 * Checks if a given request has access to delete a post.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return true|WP_Error True if the request has access to delete the item, WP_Error object otherwise.
	 */
	public function delete_item_permissions_check( $request ) {
		$check_base = parent::delete_item_permissions_check( $request );
		if ( is_wp_error( $check_base ) ) {
			return $check_base;
		}

		$group_permission_check = $this->base_permissions( $request, 'delete_item' );
		if ( is_wp_error( $group_permission_check ) ) {
			return $group_permission_check;
		}

		return true;
	}

	/**
	 * Filters the query arguments for a request.
	 *
	 * @param array           $args    Key value array of query var to query value.
	 * @param WP_REST_Request $request The request used.
	 */
	public function filter_rest_query_group_args( $args, $request ) {
		if ( isset( $request['group_id'] ) ) {
			$args['post_parent'] = (int) $request['group_id'];
		}

		return $args;
	}

	/**
	 * Map meta group connection
	 *
	 * @param WP_Post         $post     Inserted or updated post object.
	 * @param WP_REST_Request $request  Request object.
	 * @param bool            $creating True when creating a post, false when updating.
	 */
	public function set_meta_group_connection( $post, $request, $creating ) {
		if ( isset( $request['group_id'] ) ) {
			$group_id = (int) $request['group_id'];

			if ( $creating ) {
				add_post_meta( $post->ID, '_nisje_group_connection', $group_id );
			} else {
				update_post_meta( $post->ID, '_nisje_group_connection', $group_id );
			}
		}
	}

	/**
	 * Filters a wiki before it is put into db.
	 *
	 * @param stdClass        $prepared_post An object representing a single post prepare for inserting or updating the database.
	 * @param WP_REST_Request $request       Request object.
	 *
	 * @return stdClass $prepared_post An object representing a single post prepare
	 */
	public function set_group_connection( $prepared_post, $request ) {
		if ( isset( $request['group_id'] ) ) {
			$prepared_post->post_parent = (int) $request['group_id'];
		}

		return $prepared_post;
	}
}
