<?php
/**
 * REST API: WP_REST_Post_Search_Handler class
 *
 * @package Nisje
 */

/**
 * Core class representing a search handler for posts in the REST API.
 *
 * @see WP_REST_Search_Handler
 */
class User_Content_Search_Handler extends WP_REST_Search_Handler {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->type = 'user-content';
	}

	/**
	 * Searches the object type content for a given search request.
	 *
	 * @param WP_REST_Request $request Full REST request.
	 * @return array
	 */
	public function search_items( WP_REST_Request $request ) {
		$post_type_name = nisje_get_setting( 'post_type_name', 'user-content' );

		$ignore_group_ids = [];

		if ( function_exists( '\bp_is_active' ) && \bp_is_active( 'groups' ) ) {
			$groups = \groups_get_groups( [
				'status' => [ 'private', 'hidden' ],
				'fields' => 'ids',
			] );

			$user_id = bp_loggedin_user_id();

			if ( \bp_is_active( 'groups' ) ) {
				foreach ( $groups['groups'] as $group ) {
					if ( ! \groups_is_user_member( $user_id, $group ) ) {
						$ignore_group_ids[] = $group;
					}
				}
			}
		}

		$query_args = [
			'post_type'           => $post_type_name,
			'post_status'         => 'publish',
			'paged'               => (int) $request['page'],
			'posts_per_page'      => (int) $request['per_page'],
			'ignore_sticky_posts' => true,
			'fields'              => 'ids',
			'post_parent__not_in' => $ignore_group_ids,
		];

		if ( ! empty( $request['search'] ) ) {
			$query_args['s'] = $request['search'];
		}

		$query     = new WP_Query();
		$found_ids = $query->query( $query_args );
		$total     = $query->found_posts;

		return [
			self::RESULT_IDS   => $found_ids,
			self::RESULT_TOTAL => $total,
		];
	}

	/**
	 * Prepares the search result for a given ID.
	 *
	 * @param int   $id     Item ID.
	 * @param array $fields Fields to include for the item.
	 * @return array Associative array containing all fields for the item.
	 */
	public function prepare_item( $id, array $fields ) {
		$post = \nisje_get_api_data( '/wp/v2/nisje-user-content/' . $id, [
			'group_id' => get_post_meta( $id, '_nisje_group_connection', true ),
			'_fields'  => [
				'attachments',
				'author_name',
				'author',
				'comment_count',
				'content',
				'contentType',
				'date_gmt',
				'embed',
				'group_id',
				'group_name',
				'images',
				'share',
				'tags',
				'type',
			],
		] );

		unset( $post['_links'] );

		return $post;
	}

	/**
	 * Prepares links for the search result of a given ID.
	 *
	 * @param int $id Item ID.
	 * @return array Links for the given item.
	 */
	public function prepare_item_links( $id ) {
		return [];
	}
}
