<?php
/**
 * Setup Wiki Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Wiki Component Class
 */
class Wiki extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		$path = trailingslashit( dirname( __FILE__ ) );

		parent::start(
			'wiki',
			__( 'Wiki', 'nisje-wiki' ),
			$path
		);
	}

	/**
	 * Global variables
	 */
	public function setup_globals() {
		$this->field_key = 'nisje_wiki';
		$this->key       = 'di_wiki';

		$settings = get_option( 'nisje_wiki_settings' );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			foreach ( $settings as $key => $value ) {
				$this->settings[ $key ] = $value;
			}
		}

		$defaults = [
			// Public.
			'post_type_name'                  => 'dekode-wiki',
			'post_type_rest_endpoint'         => 'dekode-wiki',
			'capability_key'                  => 'di_wiki', // nisje_wiki_cap.
			// Group.
			'post_type_name_group'            => 'nisje-wiki-group',
			'post_type_rest_endpoint_group'   => 'nisje-wiki-group',
			'capability_key_group'            => 'nisje_wiki_cap_group',
			// Taxonomy.
			'taxonomy_category_name'          => 'dekode-wiki-category',
			'taxonomy_category_rest_endpoint' => 'dekode-wiki-category',
			'taxonomy_category'               => [
				'name'          => 'dekode-wiki-category',
				'rest_endpoint' => 'dekode-wiki-category',
			],
			// General.
			'field_key'                       => $this->field_key,
			'key'                             => $this->key,
		];

		$this->settings = wp_parse_args( $this->settings, $defaults );
	}

	/**
	 * Setup Filters
	 */
	public function setup_filters() {
		add_filter( 'nisje_feed_post_types', [ $this, 'add_feed_post_types' ] );
		add_filter( 'nisje_rest_shared_fields_group_post_types', [ $this, 'add_shared_rest_group_fields' ] );
		add_filter( 'nisje_rest_shared_fields_post_types', [ $this, 'add_shared_rest_fields' ] );
		add_filter( 'nisje_frontend_options', [ $this, 'add_frontend_option' ] );
		add_filter( 'nisje_rest_search_handlers', [ $this, 'add_search_handler' ] );
		add_filter( 'nisje_group_features', [ $this, 'add_group_feature' ], 50 );

		if ( is_admin() ) {
			add_action( 'nisje_settings_add_option_page', [ $this, 'add_settings_menu' ] );
			add_action( 'nisje_settings_register_field_groups', [ $this, 'register_setting_field_groups' ] );
			add_action( 'nisje_settings_populate_options_from_acf', [ $this, 'populate_acf_settings_wiki' ] );
			add_action( 'enqueue_block_editor_assets', [ $this, 'editor_assets' ], 20 );
		}
	}

	/**
	 * Editor mods
	 */
	public function editor_assets() {
		wp_enqueue_script( 'nisje-plugin-wiki-admin' );
		wp_enqueue_style( 'nisje-plugin-wiki-admin' );
	}

	/**
	 * Register Content Types
	 */
	public function register_content_types() {
		$this->include_file( 'taxonomies/taxonomy-category.php' );
		$this->include_file( 'post-types/post-type-wiki.php' );

		if ( bp_is_active( 'groups' ) ) {
			$this->include_file( 'post-types/post-type-wiki-group.php' );
		}
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		$this->include_file( 'rest-handlers/extend-wiki.php' );
		$this->include_file( 'rest-handlers/extend-groups.php' );
		$this->include_file( 'rest-handlers/class-wiki-search-handler.php' );
	}

	/**
	 * Add seach handler.
	 *
	 * @param array $handlers Handlers.
	 */
	public function add_search_handler( array $handlers ): array {
		$handlers[] = new \Wiki_Search_Handler();
		return $handlers;
	}

	/**
	 * Add wiki settings to frontend options
	 *
	 * @param array $options Options.
	 */
	public function add_frontend_option( $options ) {
		if ( \is_user_logged_in() ) {
			$user_id = \get_current_user_id();
			$user    = \get_userdata( $user_id );

			$options['wikiFeaturedBoxes']                  = nisje_get_setting( 'featured_boxes', 'wiki' );
			$options['userCapabilities']['edit']['wiki']   = $user->allcaps['edit_di_wiki'] ?? false;
			$options['userCapabilities']['create']['wiki'] = $user->allcaps['create_di_wikis'] ?? false;
		}

		return $options;
	}

	/**
	 * Add group features
	 *
	 * @param array $features Features.
	 */
	public function add_group_feature( $features ) {
		$features['enable_wiki'] = [
			'label'       => __( 'Wiki', 'wiki' ),
			'description' => 'Wiki er en enkel kunnskapsbase',
		];

		return $features;
	}

	/**
	 * Add feed post types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_feed_post_types( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_name_group', 'wiki' );
		$post_types[] = nisje_get_setting( 'post_type_name', 'wiki' );

		return $post_types;
	}

	/**
	 * Extend rest group types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_shared_rest_group_fields( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_rest_endpoint_group', 'wiki' );

		return $post_types;
	}

	/**
	 * Extend rest post types.
	 *
	 * @param array $post_types Current Post Types.
	 * @return array $post_types Populated Post Types.
	 */
	public function add_shared_rest_fields( $post_types ) {
		$post_types[] = nisje_get_setting( 'post_type_rest_endpoint_group', 'wiki' );
		$post_types[] = nisje_get_setting( 'post_type_rest_endpoint', 'wiki' );

		return $post_types;
	}

	/**
	 * Add Settings Menu
	 */
	public function add_settings_menu() {
		acf_add_options_sub_page( [
			'page_title'  => esc_html__( 'Wiki', 'nisje-wiki' ),
			'menu_title'  => esc_html__( 'Wiki', 'nisje-wiki' ),
			'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
			'menu_slug'   => 'nisje-wiki',
			'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
		] );
	}

	/**
	 * Register Setting Field Groups
	 */
	public function register_setting_field_groups() {
		$this->include_file( 'acf/field-groups/acf-field-group-options.php' );
	}

	/**
	 * Populate ACF Options Member Types
	 *
	 * @param WP_Screen $screen Screen object.
	 */
	public function populate_acf_settings_wiki( $screen ) {
		if ( false !== strpos( $screen->id, 'nisje-wiki' ) ) {
			$settings = [];

			if ( have_rows( 'wiki_featured_boxes', 'option' ) ) {
				while ( have_rows( 'wiki_featured_boxes', 'option' ) ) {
					the_row();

					$settings['featured_boxes'][] = [
						'image' => ( get_sub_field( 'image' ) ) ? get_sub_field( 'image' ) : '',
						'title' => get_sub_field( 'title' ),
						'text'  => get_sub_field( 'text' ),
						'link'  => get_sub_field( 'link' ),
					];
				}
			}

			update_option( 'nisje_wiki_settings', $settings );
		}
	}
}
