<?php
/**
 * REST API: WP_REST_Post_Search_Handler class
 *
 * @package Nisje
 */

/**
 * Core class representing a search handler for posts in the REST API.
 *
 * @see WP_REST_Search_Handler
 */
class Wiki_Search_Handler extends WP_REST_Search_Handler {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->type = 'wiki';
	}

	/**
	 * Searches the object type content for a given search request.
	 *
	 * @param WP_REST_Request $request Full REST request.
	 * @return array
	 */
	public function search_items( WP_REST_Request $request ) {
		$post_type_name = nisje_get_setting( 'post_type_name', 'wiki' );

		$query_args = [
			'post_type'           => $post_type_name,
			'post_status'         => 'publish',
			'paged'               => (int) $request['page'],
			'posts_per_page'      => (int) $request['per_page'],
			'ignore_sticky_posts' => true,
			'fields'              => 'ids',
		];

		if ( ! empty( $request['search'] ) ) {
			$query_args['s'] = $request['search'];
		}

		$query     = new WP_Query();
		$found_ids = $query->query( $query_args );
		$total     = $query->found_posts;

		return [
			self::RESULT_IDS   => $found_ids,
			self::RESULT_TOTAL => $total,
		];
	}

	/**
	 * Prepares the search result for a given ID.
	 *
	 * @param int   $id     Item ID.
	 * @param array $fields Fields to include for the item.
	 * @return array Associative array containing all fields for the item.
	 */
	public function prepare_item( $id, array $fields ) {
		$post = \nisje_get_api_data( '/wp/v2/dekode-wiki/' . $id, [
			'_fields' => [
				'breadcrumb',
				'date_gmt',
				'dekode-wiki-category',
				'excerpt',
				'title',
			],
		] );

		unset( $post['_links'] );

		return $post;
	}

	/**
	 * Prepares links for the search result of a given ID.
	 *
	 * @param int $id Item ID.
	 * @return array Links for the given item.
	 */
	public function prepare_item_links( $id ) {
		return [];
	}
}
