<?php
/**
 * Helper functions.
 *
 * @package Nisje
 */

use \Firebase\JWT\JWT;

/**
 * Helper check AUTH header.
 */
function nisje_jwt_check_auth_header() {
	$auth = filter_input( INPUT_SERVER, 'HTTP_AUTHORIZATION', FILTER_SANITIZE_STRING );
	if ( ! $auth ) {
		$auth = filter_input( INPUT_SERVER, 'REDIRECT_HTTP_AUTHORIZATION', FILTER_SANITIZE_STRING );
	}

	return $auth;
}

/**
 * Helper to validate User.
 */
function nisje_jwt_validate_user() {
	return nisje_jwt_validate_token();
}

/**
 * Helper to validate Token.
 *
 * @param bool $output Output.
 */
function nisje_jwt_validate_token( bool $output = true ) {
	$auth = nisje_jwt_check_auth_header();

	if ( ! $auth ) {
		return new WP_Error(
			'jwt_auth_no_auth_header',
			__( 'Authorization header not found.', 'nisje-jwt' ),
			[ 'status' => 403 ]
		);
	}

	list( $token ) = sscanf( $auth, 'Bearer %s' );
	if ( ! $token ) {
		return new WP_Error(
			'jwt_auth_bad_auth_header',
			__( 'Authorization header malformed.', 'nisje-jwt' ),
			[ 'status' => 403 ]
		);
	}

	/** Get the Secret Key */
	$secret_key = defined( 'JWT_AUTH_SECRET_KEY' ) ? JWT_AUTH_SECRET_KEY : false;
	if ( ! $secret_key ) {
		return new WP_Error(
			'jwt_auth_bad_config',
			__( 'JWT is not configurated properly, please contact the admin', 'nisje-jwt' ),
			[ 'status' => 403 ]
		);
	}

	try {
		$token = JWT::decode( $token, $secret_key, [ 'HS256' ] );
		if ( get_bloginfo( 'url' ) !== $token->iss ) {
			/** The iss do not match, return error */
			return new WP_Error(
				'jwt_auth_bad_iss',
				__( 'The iss do not match with this server', 'nisje-jwt' ),
				[ 'status' => 403 ]
			);
		}

		if ( ! isset( $token->data->user->id ) ) {
			return new WP_Error(
				'jwt_auth_bad_request',
				__( 'User ID not found in the token', 'nisje-jwt' ),
				[ 'status' => 403 ]
			);
		}

		if ( ! $output ) {
			return $token;
		}

		/** If the output is true return an answer to the request to show it */
		return [
			'code' => 'jwt_auth_valid_token',
			'data' => [
				'status' => 200,
			],
		];
	} catch ( Exception $e ) {
		/** Something is wrong trying to decode the token, send back the error */
		return new WP_Error(
			'jwt_auth_invalid_token',
			$e->getMessage(),
			[ 'status' => 403 ]
		);
	}
}

/**
 * Helper to generate Token.
 *
 * @param WP_Rest_Request $request Rest request.
 */
function nisje_jwt_generate_rest_token( $request ) {
	$secret_key = defined( 'JWT_AUTH_SECRET_KEY' ) ? JWT_AUTH_SECRET_KEY : false;
	$username   = $request->get_param( 'username' );
	$password   = $request->get_param( 'password' );

	/** First thing, check the secret key if not exist return a error*/
	if ( ! $secret_key ) {
		return new WP_Error(
			'jwt_auth_bad_config',
			__( 'JWT is not configurated properly, please contact the admin', 'nisje-jwt' ),
			[ 'status' => 403 ]
		);
	}

	$user = wp_authenticate( $username, $password );

	if ( is_wp_error( $user ) ) {
		$error_code = $user->get_error_code();
		return new WP_Error(
			'[jwt_auth] ' . $error_code,
			$user->get_error_message( $error_code ),
			[ 'status' => 403 ]
		);
	}

	$data = [
		'token'             => nisje_jwt_generate_token( $user ),
		'user_email'        => $user->data->user_email,
		'user_nicename'     => $user->data->user_nicename,
		'user_display_name' => $user->data->display_name,
	];

	return apply_filters( 'jwt_auth_token_before_dispatch', $data, $user );
}

/**
 * Generate token
 *
 * @param WP_User $user WordPress user.
 */
function nisje_jwt_generate_token( $user ) {
	$secret_key = defined( 'JWT_AUTH_SECRET_KEY' ) ? JWT_AUTH_SECRET_KEY : false;

	$issued_at  = time();
	$not_before = apply_filters( 'jwt_auth_not_before', $issued_at, $issued_at );
	$expire     = apply_filters( 'jwt_auth_expire', $issued_at + ( DAY_IN_SECONDS * 7 ), $issued_at );

	$token = [
		'iss'  => get_bloginfo( 'url' ),
		'iat'  => $issued_at,
		'nbf'  => $not_before,
		'exp'  => $expire,
		'data' => [
			'user' => [
				'id' => $user->data->ID,
			],
		],
	];

	$token = JWT::encode( apply_filters( 'jwt_auth_token_before_sign', $token, $user ), $secret_key );

	return $token;
}
