<?php
/**
 * Setup Main Group Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Main Group Component Class
 */
class Main_Group extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		$path = trailingslashit( dirname( __FILE__ ) );

		parent::start(
			'main-group',
			__( 'Main group', 'nisje-main-group' ),
			$path
		);
	}

	/**
	 * Includes
	 */
	public function includes() {
		$this->include_file( 'helper-functions.php' );
	}

	/**
	 * Setup Filters
	 */
	public function setup_filters() {
		add_filter( 'nisje_custom_profile_layouts', [ $this, 'register_profile_layout' ] );
		add_filter( 'nisje_custom_profile_layout_value_main_group', [ $this, 'register_profile_layout_value' ], 10, 2 );
		add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_group' ] );

		// Autojoin user to main group when updating/creating user.
		add_action( 'user_register', [ $this, 'autojoin' ] );
		add_action( 'personal_options_update', [ $this, 'autojoin' ] );
		add_action( 'edit_user_profile_update', [ $this, 'autojoin' ] );
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		$this->include_file( 'rest-handlers/extend-user.php' );
	}

	/**
	 * Register Field groups.
	 */
	public function register_field_group() {
		$this->include_file( 'acf/field-groups/acf-field-group-main-group.php' );
	}

	/**
	 * Register profile layout.
	 *
	 * @param array $layouts Registered profile layouts.
	 * @return array $layouts Populated profile layouts.
	 */
	public function register_profile_layout( $layouts ) {
		$layouts['main_group'] = esc_html__( 'Main Group', 'nisje' );

		return $layouts;
	}

	/**
	 * Profile layout value.
	 *
	 * @param array $field   Profile field.
	 * @param array $user_id User ID.
	 *
	 * @return array Array with profile values.
	 */
	public function register_profile_layout_value( $field, $user_id ) {
		if ( 'main_group' === $field['field'] ) {
			$main_group_id = nisje_get_main_group( $user_id );

			$group = groups_get_group( [
				'group_id' => $main_group_id,
			] );

			if ( ! $group->id ) {
				return '';
			}

			$prepend = isset( $field['prepend'] ) ? esc_html( $field['prepend'] ) . ' ' : '';

			$retval = [
				'main_group_id' => $main_group_id,
				'group_name'    => $prepend . bp_get_group_name( $group ),
			];

			return [
				'name'  => esc_html__( 'Main group', 'nisje' ),
				'value' => $retval,
				'type'  => 'main-group',
			];
		}
	}

	/**
	 * Autojoin user to main group when saving user profile.
	 *
	 * @param int $user_id User ID.
	 */
	public function autojoin( $user_id ) {
		if ( ! is_admin() ) {
			return;
		}

		// Auto add user to his/her main group.
		if ( current_user_can( 'edit_user', $user_id ) && isset( $_POST['acf']['di_key_main_group'] ) ) { // phpcs:ignore
			$group_id = absint( $_POST['acf']['di_key_main_group'] ); // phpcs:ignore
			if ( ! groups_is_user_member( $user_id, $group_id ) ) {
				groups_join_group( $group_id, $user_id );
			}
		}
	}
}
