<?php
/**
 * Setup Biblioteksutvikling Library component
 *
 * @package Nisje
 */

declare( strict_types = 1 );
namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

if ( ! class_exists( '\\Dekode\\Nisje\\Components\\NB_Library' ) && class_exists( '\\Dekode\\Nisje\\Components\\Component' ) ) {
	/**
	 * Library component class
	 */
	class NB_Library extends Component {
		/**
		 * Helper to make current library type global.
		 *
		 * @var string $current_library_type
		 */
		private $current_library_type = '';

		/**
		 * Helper to make library tax query global.
		 *
		 * @var string $library_tax_query
		 */
		private $library_tax_query = '';

		/**
		 * Constructor
		 */
		public function __construct() {
			$path = trailingslashit( __DIR__ );

			parent::start(
				'nb-library',
				esc_html__( 'NB Library', 'nisje-nb' ),
				$path
			);
		}

		/**
		 * Global variables
		 */
		public function setup_globals() {
			$defaults = [
				// Taxonomy.
				'taxonomy_type_name'         => 'nisje-nb-library-type',
				'taxonomy_municipality_name' => 'nisje-nb-library-municipality',
				'taxonomy_county_name'       => 'nisje-nb-library-county',
			];

			$this->settings = wp_parse_args( $this->settings, $defaults );
		}

		/**
		 * Register public settings.
		 */
		public function register_public_settings() {
			$this->public_settings = [
				'types' => nisje_nb_map_library_types( 'all' ),
			];
		}

		/**
		 * Setup filters
		 */
		public function setup_filters() {
			add_filter( 'nisje_register_group_types', [ $this, 'add_library_group_type' ] );
			add_action( 'bp_send_email', [ $this, 'membership_request_notification' ], 10, 4 );

			// Support filtering groups/libraries by library_type.
			add_filter( 'nisje_rest_groups_query', [ $this, 'parse_groups_query' ], 10, 2 );
			add_filter( 'bp_groups_get_paged_groups_sql', [ $this, 'parse_select' ], 10, 2 );
			add_filter( 'bp_groups_get_total_groups_sql', [ $this, 'parse_total' ], 10, 2 );
		}

		/**
		 * Register content types
		 */
		public function register_content_types() {
			// Taxonomies.
			$this->include_file( 'taxonomies/taxonomy-nb-library-type.php' );
			$this->include_file( 'taxonomies/taxonomy-nb-library-municipality.php' );
			$this->include_file( 'taxonomies/taxonomy-nb-library-county.php' );
		}

		/**
		 * Register REST Endpoints
		 */
		public function register_rest_routes() {
			$this->include_file( 'rest-handlers/extend-user.php' );
			$this->include_file( 'rest-handlers/extend-groups.php' );
		}

		/**
		 * Add library group type.
		 *
		 * @param array $group_types Array of registered group_types.
		 * @return array $group_types Array of registered group_types.
		 */
		public function add_library_group_type( $group_types ) {
			$group_types['library'] = [
				'key'                   => 'library',
				'labels'                => [
					'name'          => esc_html__( 'Libraries', 'nisje-nb' ),
					'singular_name' => esc_html__( 'Library', 'nisje-nb' ),
				],
				'directory_slug'        => 'library',
				'has_directory'         => 'library',
				'show_in_create_screen' => false,
				'show_in_list'          => false,
				'exclude_groups'        => true,
				'create_screen_checked' => false,
				'description'           => '',
			];

			$group_types['schoollibrary'] = [
				'key'                   => 'schoollibrary',
				'labels'                => [
					'name'          => esc_html__( 'School Libraries', 'nisje-nb' ),
					'singular_name' => esc_html__( 'School Library', 'nisje-nb' ),
				],
				'directory_slug'        => 'schoollibrary',
				'has_directory'         => 'schoollibrary',
				'show_in_create_screen' => false,
				'show_in_list'          => false,
				'exclude_groups'        => true,
				'create_screen_checked' => false,
				'description'           => '',
			];

			$group_types['municipality'] = [
				'key'                   => 'municipality',
				'labels'                => [
					'name'          => esc_html__( 'Municipalities', 'nisje-nb' ),
					'singular_name' => esc_html__( 'Municipality', 'nisje-nb' ),
				],
				'directory_slug'        => 'municipality',
				'has_directory'         => 'municipality',
				'show_in_create_screen' => false,
				'show_in_list'          => false,
				'exclude_groups'        => true,
				'create_screen_checked' => false,
				'description'           => '',
			];

			$group_types['countycouncil'] = [
				'key'                   => 'countycouncil',
				'labels'                => [
					'name'          => esc_html__( 'County council', 'nisje-nb' ),
					'singular_name' => esc_html__( 'County council', 'nisje-nb' ),
				],
				'directory_slug'        => 'countycouncil',
				'has_directory'         => 'countycouncil',
				'show_in_create_screen' => false,
				'show_in_list'          => false,
				'exclude_groups'        => true,
				'create_screen_checked' => false,
				'description'           => '',
			];

			return $group_types;
		}

		/**
		 * Add library leader to membership-request notification
		 *
		 * @param BP_Email $email E-mail.
		 * @param string   $type  E-mail type.
		 * @param string   $to    E-mail receiver.
		 * @param array    $args  Arguments.
		 */
		public function membership_request_notification( &$email, $type, $to, $args ) {
			if ( 'groups-membership-request' === $type ) {
				$args['tokens']['library-leader'] = '';

				if ( isset( $args['tokens']['requesting-user.id'], $args['tokens']['group.id'] ) ) {
					$requesting_id = (int) $args['tokens']['requesting-user.id'];
					$group_id      = (int) $args['tokens']['group.id'];

					$leader  = xprofile_get_field_data( 6, $requesting_id );
					$library = (int) xprofile_get_field_data( 5, $requesting_id );
					if ( 'Ja' === $leader && $library ) {
						if ( $group_id === $library ) {
							$args['tokens']['library-leader'] = esc_html__( 'This is the leader of the library', 'nisje-nb' );
						}
					}
				}

				$email->set_tokens( $args['tokens'] );
			}
		}

		/**
		 * Groups query.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request used.
		 */
		public function parse_groups_query( $args, $request ) {
			$library_type = $request->get_param( 'library_type' );
			if ( $library_type ) {
				$this->current_library_type = $library_type;
			}

			return $args;
		}

		/**
		 * Build a WP_Tax_Query if a library type isset.
		 *
		 * @param array $sql_parts Current SQL parts.
		 * @return array Optimized SQL parts with current library type.
		 */
		public function get_sql_parts( array $sql_parts = [] ) : array {
			if ( empty( $this->current_library_type ) ) {
				return $sql_parts;
			}

			$slugs = nisje_nb_get_library_type_slugs( $this->current_library_type );
			if ( ! $slugs ) {
				return $sql_parts;
			}

			$tax_query = new \WP_Tax_Query( [
				[
					'taxonomy' => nisje_get_setting( 'taxonomy_type_name', 'nb_library' ),
					'terms'    => $slugs,
					'field'    => 'slug',
				],
			] );

			$this->library_tax_query = $tax_query->get_sql( 'g', 'id' );

			$where = $this->library_tax_query['where'];

			/**
			 * This can be the case when filtering alphabetically
			 * or by created date.
			 */
			if ( empty( $sql_parts['where'] ) ) {
				$where_parts = explode( 'AND', $where );

				if ( count( $where_parts ) > 1 ) {
					// Remove the first 'AND'.
					array_shift( $where_parts );
				}
				$where = sprintf( 'WHERE %s', join( 'AND', $where_parts ) );
			} else {
				$where = sprintf( 'WHERE %1$s%2$s', $sql_parts['where'], $this->library_tax_query['where'] );
			}

			$sql_parts['from']  = sprintf( 'FROM %1$s%2$s', $sql_parts['from'], $this->library_tax_query['join'] );
			$sql_parts['where'] = $where;

			return $sql_parts;
		}

		/**
		 * Parse group select query if library type isset.
		 *
		 * @param string $query Current query.
		 * @param array  $sql_parts Current SQL parts.
		 *
		 * @return string Optimized query with library type SQL.
		 */
		public function parse_select( string $query = '', array $sql_parts = [] ) : string {
			if ( ! empty( $this->current_library_type ) ) {
				$query = join( ' ', $this->get_sql_parts( $sql_parts ) );
			}

			return $query;
		}

		/**
		 * Parse group total query if library type isset.
		 *
		 * @param string $query     Current query.
		 * @param array  $sql_parts Current SQL parts.
		 *
		 * @return string Optimized query with library type SQL.
		 */
		public function parse_total( string $query = '', array $sql_parts = [] ) : string {
			if ( ! empty( $this->current_library_type ) && ! empty( $this->library_tax_query ) ) {
				$sql = array_merge(
					[
						'select' => 'SELECT COUNT(DISTINCT g.id)',
					],
					array_intersect_key( $this->get_sql_parts( $sql_parts ), [
						'from'  => true,
						'where' => true,
					] )
				);

				$query = join( ' ', $sql );
			}

			return $query;
		}
	}
}
