<?php
/**
 * Setup Settings Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Settings Component Class
 */
class Settings extends Component {

	/**
	 * Constructor
	 */
	public function __construct() {
		$path = trailingslashit( dirname( __FILE__ ) );

		parent::start(
			'settings',
			esc_html__( 'Settings', 'nisje' ),
			$path,
			'general',
			'nisje_settings_general'
		);
	}

	/**
	 * Setup globals
	 */
	public function setup_globals() {
		$settings = get_option( $this->option_key );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			foreach ( $settings as $key => $value ) {
				$this->settings[ $key ] = $value;
			}
		}

		$defaults = [
			'name'                     => $this->name,
			'option_key'               => $this->option_key,
			'field_key_prefix'         => 'nisje_settings',
			// Admin settings.
			'exclude_members'          => [],
			'skip_second_registration' => false,
			'hide_forgotten_password'  => false,
			'terms'                    => false,
			'login_info_message'       => false,
			'mail_from'                => '',
			'mail_from_name'           => '',
			'community_id'             => '',
			'tracking_id'              => '',
		];

		$this->settings = array_merge( $defaults, (array) $this->settings );
	}

	/**
	 * Setup Filters
	 */
	public function setup_filters() {
		if ( is_admin() ) {
			add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_group' ] );
			add_action( 'nisje_settings_populate_options_from_acf', [ $this, 'populate_acf_settings' ] );
		}
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		$this->include_file( 'rest-handlers/class-settings-controller.php' );

		$controller = new \Dekode\Nisje\Components\Rest\Settings_Controller();
		$controller->register_routes();
	}

	/**
	 * Include acf field groups.
	 */
	public function register_field_group() {
		$this->include_file( 'acf/field-groups/acf-field-group-options-general.php' );
	}

	/**
	 * Populate ACF Options General.
	 *
	 * @param WP_Screen $screen Screen object.
	 */
	public function populate_acf_settings( $screen ) {
		if ( 'toplevel_page_nisje' === $screen->id ) {
			$field_key_prefix = nisje_get_setting( 'field_key_prefix', 'general' );

			$current_settings = nisje_get_setting( 'general' );

			$terms = get_field( $field_key_prefix . '_terms', 'option' );
			if ( empty( $terms ) ) {
				$terms = false;
			}

			$login_info_message = get_field( $field_key_prefix . '_login_info_message', 'option' );
			if ( empty( $login_info_message ) ) {
				$login_info_message = false;
			}

			$acf_settings = [
				'terms'                    => $terms,
				'login_info_message'       => $login_info_message,
				'mail_from'                => get_field( $field_key_prefix . '_mail_from', 'option' ),
				'mail_from_name'           => get_field( $field_key_prefix . '_mail_from_name', 'option' ),
				'community_id'             => get_field( $field_key_prefix . '_community_id', 'option' ),
				'tracking_id'              => get_field( $field_key_prefix . '_tracking_id', 'option' ),
				'skip_second_registration' => get_field( $field_key_prefix . '_skip_second_registration', 'option' ),
				'hide_forgotten_password'  => get_field( $field_key_prefix . '_hide_forgotten_password', 'option' ),
			];

			$exclude_members = get_field( $field_key_prefix . '_exclude_members', 'option' );
			if ( $exclude_members ) {
				foreach ( $exclude_members as $exclude_member ) {
					$acf_settings['exclude_members'][] = $exclude_member['ID'];

					$groups = groups_get_user_groups( $exclude_member['ID'] );
					if ( isset( $groups['groups'] ) ) {
						foreach ( $groups['groups'] as $group ) {
							if ( count( groups_get_group_admins( $group ) ) < 2 && groups_is_user_admin( $exclude_member['ID'], $group ) ) {
								continue;
							} else {
								$response = ( groups_remove_member( $exclude_member['ID'], $group ) ) ? 'Removed' : 'Not removed';
							}
						}
					}
				}
			}

			$settings = array_merge( $current_settings, $acf_settings );

			update_option( $this->option_key, $settings );
		}
	}
}
