<?php
/**
 * Avatar helper functions.
 *
 * @package Nisje
 */

/**
 * Get user avatar
 *
 * @param int $user_id User ID.
 * @return array User Avatars
 */
function nisje_get_user_avatar( $user_id ) {
	$cache_key = 'nisje_' . $user_id . '_user_avatar';

	$retval = wp_cache_get( $cache_key, 'avatars' );

	if ( false === $retval ) {
		$avatar = new Nisje_Avatar( $user_id, 'user' );
		$retval = [
			'src'          => $avatar->src,
			'default_src'  => $avatar->default_src,
			'gravatar_src' => [
				'full'   => bp_core_fetch_avatar( [
					'html'    => false,
					'object'  => 'user',
					'item_id' => $user_id,
					'type'    => 'full',
				] ),
				'medium' => bp_core_fetch_avatar( [
					'html'    => false,
					'object'  => 'user',
					'item_id' => $user_id,
					'type'    => 'medium',
				] ),
				'thumb'  => bp_core_fetch_avatar( [
					'html'    => false,
					'object'  => 'user',
					'item_id' => $user_id,
					'type'    => 'thumb',
				] ),
			],
			'name'         => bp_core_get_user_displayname( $user_id ),
		];

		wp_cache_set( $cache_key, $retval, 'avatars', 3 * HOUR_IN_SECONDS );
	}

	return $retval;
}

/**
 * Get group avatar
 *
 * @param int $group_id Group ID.
 * @return array Group Avatars
 */
function nisje_get_group_avatar( $group_id ) {
	$cache_key = 'nisje_' . $group_id . '_group_avatar';

	$retval = wp_cache_get( $cache_key, 'avatars' );

	if ( false === $retval ) {
		$avatar = new Nisje_Avatar( $group_id, 'group' );

		$group = groups_get_group( [
			'group_id' => (int) $group_id,
		] );

		$name  = '';
		$color = 0;
		if ( $group->id ) {
			$name  = $group->name;
			$color = (int) groups_get_groupmeta( $group->id, 'color' );
		}

		$retval = [
			'src'          => $avatar->src,
			'default_src'  => $avatar->default_src,
			'gravatar_src' => [
				'full'   => bp_core_fetch_avatar( [
					'html'    => false,
					'object'  => 'group',
					'item_id' => $group_id,
					'type'    => 'full',
				] ),
				'medium' => bp_core_fetch_avatar( [
					'html'    => false,
					'object'  => 'group',
					'item_id' => $group_id,
					'type'    => 'medium',
				] ),
				'thumb'  => bp_core_fetch_avatar( [
					'html'    => false,
					'object'  => 'group',
					'item_id' => $group_id,
					'type'    => 'full',
				] ),
			],
			'name'         => $name,
			'color'        => $color,
		];

		wp_cache_set( $cache_key, $retval, 'avatars', 3 * HOUR_IN_SECONDS );
	}

	return $retval;
}

/**
 * Default avatar image sizes.
 */
function nisje_get_avatar_sizes() {
	$sizes = [
		'thumb'  => [
			'width'  => 50,
			'height' => 50,
		],
		'medium' => [
			'width'  => 150,
			'height' => 150,
		],
		'full'   => [
			'width'  => 300,
			'height' => 300,
		],
	];

	return apply_filters( 'nisje_avatar_sizes', $sizes );
}

/**
 * Avatar Rest Properties.
 */
function nisje_get_avatar_rest_properties() {
	$sizes = nisje_get_avatar_sizes();

	$type_properties = [
		'src',
		'default_src',
		'gravatar_src',
		'name',
	];

	$size_properties = [];
	foreach ( $sizes as $key => $size ) {
		$size_properties[ $key ] = [
			'description' => esc_html__( 'Size avatar URL', 'nisje' ),
			'type'        => 'string',
			'format'      => 'uri',
			'context'     => [ 'view', 'edit', 'embed' ],
		];
	}

	return $size_properties;
}
