<?php
/**
 * Main Nisje Class.
 *
 * @package Nisje
 */

namespace Dekode\Nisje;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Nisje Core Class.
 */
class Core {

	/**
	 * Settings API.
	 *
	 * @var object
	 */
	public $settings;

	/**
	 * Plugin path
	 *
	 * @var string
	 */
	public $path;

	/**
	 * Plugin dir
	 *
	 * @var string
	 */
	public $dir;

	/**
	 * Versions.
	 *
	 * @var array
	 */
	public $versions;

	/**
	 * Components.
	 *
	 * @var array
	 */
	public $components;

	/**
	 * A dummy constructor
	 */
	public function __construct() {
		// Do nothing.
	}

	/**
	 * Main Instance
	 *
	 * @static object $instance
	 * @return Nisje|null The object.
	 */
	public static function instance() {
		// Store the instance locally to avoid private static replication.
		static $instance = null;

		// Only run these methods if they haven't been run previously.
		if ( null === $instance ) {
			$instance = new Core();

			$instance->setup_globals();
			$instance->includes();
			$instance->setup_actions();
		}

		// Always return the instance.
		return $instance;
	}

	/**
	 * Global variables
	 */
	private function setup_globals() {
		global $wpdb;

		$this->dir  = plugin_dir_url( __FILE__ );
		$this->path = plugin_dir_path( __FILE__ );

		$this->versions = [
			'plugin_version' => NISJE_CORE_VERSION,
			'db_version'     => NISJE_CORE_DB_VERSION,
		];

		$wpdb->di_event_attending = $wpdb->prefix . 'di_event_attending';
	}

	/**
	 * Include required files
	 */
	private function includes() {
		nisje_include( 'includes/global-update-functions.php' );
		nisje_include( 'includes/global-rest-functions.php' );

		// Setup and initialize settings class.
		nisje_include( 'includes/class-settings.php' );
		$this->settings = Settings::instance();

		// Normalize/optimize the installation.
		nisje_include( 'includes/resets/resets.php' );

		// Include rest helper.
		nisje_include( 'includes/helpers/class-wp-api-helper.php' );
	}

	/**
	 * Set up the default hooks and actions
	 */
	private function setup_actions() {
		if ( is_admin() ) {
			// Admin requirements and notifications.
			add_action( 'admin_init', [ $this, 'requirements' ] );
			add_action( 'admin_init', [ $this, 'updates' ] );
		}

		// Load text domain on plugins_loaded.
		add_action( 'plugins_loaded', [ $this, 'load_textdomain' ] );

		// Add custom hook to help sub modules.
		add_action( 'plugins_loaded', [ $this, 'include_hook' ], 11 );

		// Core functionality.
		add_action( 'nisje_include', [ $this, 'core_functionality' ], 100 );

		// Include components.
		add_action( 'nisje_include', [ $this, 'components' ], 100 );

		// Register taxonomies, post_types and metadata.
		add_action( 'init', [ $this, 'register_content_types' ] );

		// Register ACF data.
		add_action( 'plugins_loaded', [ $this, 'register_acf' ], 100 );

		// Register Rest handlers.
		add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );

		// Close down API. Users must be logged in to use it!
		add_filter( 'rest_authentication_errors', [ $this, 'filter_routes' ] );

		// Addional rest headers.
		add_action( 'rest_pre_serve_request', [ $this, 'additional_rest_headers' ] );

		// Register Theme Functionality.
		add_action( 'after_setup_theme', [ $this, 'register_theme_functionality' ] );

		add_action( 'admin_init', [ $this, 'restrict_admin' ], 1 );
	}

	/**
	 * Validate whether the requirements for this plugin are available.
	 */
	public function requirements() {
		if ( ! is_plugin_active( 'buddypress/bp-loader.php' ) || ! is_plugin_active( 'advanced-custom-fields-pro/acf.php' ) ) {
			add_action( 'admin_notices', function() {
				?>
				<div class="error">
					<p>
						<strong><?php echo esc_html( nisje_get_setting( 'name', 'core' ) ); ?>:</strong>
						<?php esc_html_e( 'BuddyPress or ACF Pro is missing. Please activate BuddyPress and ACF before you activate this plugin.', 'nisje' ); ?>
					</p>
				</div>
				<?php
			} );

			deactivate_plugins( plugin_basename( __FILE__ ) );
		}
	}

	/**
	 * Check if plugin needs re-activation.
	 */
	public function updates() {
		do_action( 'nisje_updates' );

		$current_plugin_version = nisje_get_version();
		$installed_version      = get_option( 'nisje_version' );

		// New install.
		if ( ! $installed_version ) {
			update_option( 'nisje_version', $current_plugin_version );

			nisje_activation();

			return;
		}

		// Do nothing if $installed_version is >= $plugin_version.
		if ( version_compare( $installed_version, $current_plugin_version, '>=' ) ) {
			return;
		} else {
			nisje_activation();

			add_action( 'admin_notices', function () {
				?>
				<div class="notice">
					<p>
						<strong><?php echo esc_html( nisje_get_setting( 'name', 'core' ) ); ?>:</strong>
						<?php esc_html_e( 'Plugin updated!', 'nisje' ); ?>
					</p>
				</div>
				<?php
			} );
		}
	}

	/**
	 * Nisje Include Hook
	 */
	public function include_hook() {
		if ( function_exists( '\buddypress' ) ) {
			do_action( 'nisje_include' );
		}
	}

	/**
	 * Load Plugin Text Domain
	 */
	public function load_textdomain() {
		load_plugin_textdomain( 'nisje', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
	}

	/**
	 * Include core functionality
	 */
	public function core_functionality() {
		nisje_include( 'includes/core/comments/comments.php' );
		nisje_include( 'includes/core/groups/groups.php' );
		nisje_include( 'includes/core/mentions/mentions.php' );
		nisje_include( 'includes/core/notifications/notifications.php' );
		nisje_include( 'includes/core/reactions/reactions.php' );
		nisje_include( 'includes/core/user/user.php' );
	}

	/**
	 * Include Components
	 */
	public function components() {
		nisje_include( 'includes/components/class-component.php' );

		$this->include_core_components();

		do_action( 'nisje_register_components' );

		$registered_components = apply_filters( 'nisje_components', [] );

		foreach ( (array) $registered_components as $component ) {
			if ( ! ( $component instanceof \Dekode\Nisje\Components\Component ) ) {
				$component = new $component();
			}

			// Magically do includes.
			if ( method_exists( $component, 'includes' ) ) {
				$component->includes();
			}

			// Magically setup globals.
			if ( method_exists( $component, 'setup_globals' ) ) {
				$component->setup_globals();
			}

			// Magically setup filters.
			if ( method_exists( $component, 'setup_filters' ) ) {
				$component->setup_filters();
			}

			$this->components[ $component->name ] = $component;
		}

		do_action( 'nisje_components_registered' );
	}

	/**
	 * Include Core Components
	 */
	private function include_core_components() {
		nisje_include( 'includes/components/attachments/attachments.php' );
		nisje_include( 'includes/components/avatars/avatars.php' );
		nisje_include( 'includes/components/customizer/customizer.php' );
		nisje_include( 'includes/components/mail-notifications/mail-notifications.php' );
		nisje_include( 'includes/components/feed/feed.php' );
		nisje_include( 'includes/components/group-types/group-types.php' );
		nisje_include( 'includes/components/groups/groups.php' );
		nisje_include( 'includes/components/member-types/member-types.php' );
		nisje_include( 'includes/components/profile-layouts/profile-layouts.php' );
		nisje_include( 'includes/components/reaction/reaction.php' );
		nisje_include( 'includes/components/scraper/scraper.php' );
		nisje_include( 'includes/components/settings/settings.php' );
		nisje_include( 'includes/components/xprofile/xprofile.php' );
	}

	/**
	 * Restrict Backend
	 */
	public function restrict_admin() {
		if ( is_admin() && ! current_user_can( 'moderate_comments' ) && ! ( defined( 'DOING_AJAX' ) && DOING_AJAX ) ) {
			wp_safe_redirect( home_url( '/' ) );
			exit;
		}
	}

	/**
	 * Register Content Types
	 */
	public function register_content_types() {
		do_action( 'nisje_register_taxonomies', $this->versions['plugin_version'] );
		do_action( 'nisje_register_post_types', $this->versions['plugin_version'] );
		do_action( 'nisje_register_metadata', $this->versions['plugin_version'] );
		do_action( 'nisje_register_content_types', $this->versions['plugin_version'] );

		nisje_include( 'includes/taxonomies/taxonomy-associated.php' );
		nisje_include( 'includes/taxonomies/taxonomy-access.php' );
		nisje_include( 'includes/taxonomies/taxonomy-map-group-types.php' );
		nisje_include( 'includes/taxonomies/taxonomy-tags.php' );
	}

	/**
	 * Register ACF Functionality
	 */
	public function register_acf() {
		// Core ACF.
		add_action( 'acf/include_fields', [ $this, 'register_acf_fields' ] );
		add_action( 'acf/include_field_types', [ $this, 'register_acf_field_types' ] );

		// 3rd party ACF.
		do_action( 'nisje_register_acf', $this->versions['plugin_version'] );
	}

	/**
	 * Register acf fields
	 */
	public function register_acf_fields() {
		do_action( 'nisje_register_acf_fields', $this->versions['plugin_version'] );
	}

	/**
	 * Register acf fields
	 */
	public function register_acf_field_types() {
		nisje_include( 'includes/acf/fields/class-acf-field-bp-groups.php' );
		nisje_include( 'includes/acf/fields/class-acf-field-bp-member-types.php' );
		nisje_include( 'includes/acf/fields/class-acf-field-bp-profile-fields.php' );
		nisje_include( 'includes/acf/fields/class-acf-field-bp-user-groups.php' );

		// 3rd party ACF fields
		do_action( 'nisje_register_acf_field_types', $this->versions['plugin_version'] );
	}

	/**
	 * Register Rest routes
	 */
	public function register_rest_routes() {
		global $wp_query;

		if ( ! class_exists( 'WP_REST_Controller' ) ) {
			return;
		}

		// Include global/shared rest handlers.
		nisje_include( 'includes/rest-handlers/class-base-group-post-controller.php' );
		nisje_include( 'includes/rest-handlers/class-base-archive-post-controller.php' );

		// Extend group post-types with shared data.
		nisje_include( 'includes/rest-handlers/extend-group-post-types.php' );
		// Extend post-types with shared data.
		nisje_include( 'includes/rest-handlers/extend-post-types.php' );

		do_action( 'nisje_register_rest_routes' );
	}

	/**
	 * Only allow certain REST endpoints.
	 *
	 * @param WP_Error|null|bool $result WP_Error if authentication error, null if authentication method wasn't used, true if authentication succeeded.
	 * @return WP_Error|null|bool WP_Error if authentication error, null if authentication method wasn't used, true if authentication succeeded.
	 */
	public function filter_routes( $result ) {
		global $wp;

		if ( is_user_logged_in() ) {
			return $result;
		}

		$current_route = $wp->query_vars['rest_route'];
		$current_route = ( empty( $current_route ) || '/' === $current_route ) ? $current_route : untrailingslashit( $current_route );

		$open_routes = [
			'/nisje/v1/settings',
			'/dekode-intranet/v1/users',
			'/wp/v2/users',
			'/jwt-auth/v1',
			'/jwt-auth/v1/token',
			'/jwt-auth/v1/token/validate',
		];

		$open_routes = apply_filters( 'nisje_filter_routes', $open_routes );

		if ( ! in_array( $current_route, $open_routes, true ) ) {
			return new \WP_Error( 'rest_not_logged_in', esc_html__( 'You are not currently logged in.', 'nisje' ), [
				'status' => 401,
			] );
		}

		return $result;
	}

	/**
	 * Add 'Platform -> nisje' header to all responses.
	 */
	public function additional_rest_headers() {
		header( 'Platform: nisje' );

		do_action( 'nisje_additional_rest_headers' );
	}

	/**
	 * Register Theme Functionality
	 */
	public function register_theme_functionality() {
		// Register default Menu.
		register_nav_menu( 'primary', esc_html__( 'Primary Menu', 'nisje' ) );

		// Add post thumbnails.
		add_theme_support( 'post-thumbnails' );

		// Enable support for custom logo.
		add_theme_support( 'custom-logo', [
			'height'      => 240,
			'width'       => 240,
			'flex-height' => true,
			'flex-width'  => true,
		] );

		add_theme_support( 'custom-background', [
			'default-color' => 'f1f1f1',
		] );

		// Indicate widget sidebars can use selective refresh in the Customizer.
		add_theme_support( 'customize-selective-refresh-widgets' );

		do_action( 'nisje_register_theme_functionality' );
	}
}
